<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bed;
use App\Models\Floor;
use App\Models\Room;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class BedController extends Controller
{
    public function index()
    {
        $beds = Bed::query()->with(['room.floor'])->orderBy('room_id')->orderBy('name')->get();
        return view('admin.master.beds.index', compact('beds'));
    }

    public function show(Bed $bed)
    {
        $bed->load(['room.floor']);
        return view('admin.master.beds.show', compact('bed'));
    }

    public function create(Request $request)
    {
        $floors = Floor::query()->with('rooms')->orderBy('sort_order')->orderBy('name')->get();
        $rooms = Room::query()->orderBy('floor_id')->orderBy('name')->get();
        return view('admin.master.beds.create', compact('floors', 'rooms'));
    }

    public function store(Request $request)
    {
        $mode = $request->input('mode', 'single'); // single | bulk

        if ($mode === 'bulk') {
            $validated = $request->validate([
                'room_id' => ['required', 'exists:rooms,id'],
                'count' => ['required', 'integer', 'min:1', 'max:50'],
                'name_prefix' => ['nullable', 'string', 'max:100'],
                'bulk_rent_amount' => ['required', 'numeric', 'min:0'],
            ]);
            $room = Room::with('floor')->findOrFail($validated['room_id']);
            $currentBeds = $room->beds()->count();
            $capacity = (int) $room->capacity;
            $count = (int) $validated['count'];
            if ($capacity > 0 && ($currentBeds + $count) > $capacity) {
                $allowed = $capacity - $currentBeds;
                return back()->withInput()->withErrors([
                    'count' => "Room capacity is {$capacity} and already has {$currentBeds} bed(s). You can add at most {$allowed} more.",
                ]);
            }
            $floor = $room->floor;
            $floorBedsCount = (int) $floor->rooms()->withCount('beds')->get()->sum('beds_count');
            if ($floor->capacity > 0 && ($floorBedsCount + $count) > $floor->capacity) {
                $allowed = $floor->capacity - $floorBedsCount;
                return back()->withInput()->withErrors([
                    'count' => "Floor capacity is {$floor->capacity}. You can add at most {$allowed} more bed(s) on this floor.",
                ]);
            }
            $prefix = trim($validated['name_prefix'] ?? 'Bed');
            $created = 0;
            $existingNames = $room->beds()->pluck('name')->map(fn ($n) => $n)->flip();
            for ($i = 1; $i <= $count; $i++) {
                $name = $prefix . ' ' . $i;
                if ($existingNames->has($name)) {
                    continue;
                }
                Bed::query()->create([
                    'room_id' => $room->id,
                    'name' => $name,
                    'status' => Bed::STATUS_AVAILABLE,
                    'rent_amount' => $validated['bulk_rent_amount'],
                ]);
                $created++;
                $existingNames[$name] = true;
            }
            $message = $created === (int) $validated['count']
                ? "{$created} bed(s) added successfully."
                : "{$created} bed(s) added. Some names were skipped (already exist).";
            return redirect()->route('admin.beds.index')->with('success', $message);
        }

        // single
        $validated = $request->validate([
            'room_id' => ['required', 'exists:rooms,id'],
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('beds', 'name')->where('room_id', $request->room_id),
            ],
            'rent_amount' => ['required', 'numeric', 'min:0'],
        ]);

        $room = Room::with('floor')->findOrFail($validated['room_id']);
        $currentBeds = $room->beds()->count();
        $capacity = (int) $room->capacity;
        if ($capacity > 0 && ($currentBeds + 1) > $capacity) {
            return back()->withInput()->withErrors([
                'room_id' => "Room capacity is {$capacity} and already has {$currentBeds} bed(s). Cannot add more beds to this room.",
            ]);
        }
        $floor = $room->floor;
        $floorBedsCount = (int) $floor->rooms()->withCount('beds')->get()->sum('beds_count');
        if ($floor->capacity > 0 && ($floorBedsCount + 1) > $floor->capacity) {
            return back()->withInput()->withErrors([
                'room_id' => "Floor capacity is {$floor->capacity}. Cannot add more beds to this floor.",
            ]);
        }

        Bed::query()->create([
            'room_id' => $validated['room_id'],
            'name' => $validated['name'],
            'status' => Bed::STATUS_AVAILABLE,
            'rent_amount' => $validated['rent_amount'],
        ]);

        return redirect()
            ->route('admin.beds.index')
            ->with('success', 'Bed added successfully.');
    }

    public function edit(Bed $bed)
    {
        $bed->load('room.floor');
        $floors = Floor::query()->with('rooms')->orderBy('sort_order')->orderBy('name')->get();
        $rooms = Room::query()->orderBy('floor_id')->orderBy('name')->get();
        return view('admin.master.beds.edit', compact('bed', 'floors', 'rooms'));
    }

    public function update(Request $request, Bed $bed)
    {
        $validated = $request->validate([
            'room_id' => ['required', 'exists:rooms,id'],
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('beds', 'name')->where('room_id', $request->room_id)->ignore($bed->id),
            ],
            'rent_amount' => ['required', 'numeric', 'min:0'],
            'status' => ['required', 'in:available,occupied'],
        ]);
        $bed->update($validated);
        return redirect()->route('admin.beds.index')->with('success', 'Bed updated successfully.');
    }

    public function destroy(Bed $bed)
    {
        $bed->delete();
        return redirect()
            ->route('admin.beds.index')
            ->with('success', 'Bed deleted successfully.');
    }
}
