<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BankEntry;
use App\Models\Bed;
use App\Models\Deposit;
use App\Models\Drawing;
use App\Models\Expense;
use App\Models\Floor;
use App\Models\Rent;
use App\Models\Student;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class DashboardController extends Controller
{
    /**
     * Compute cash vs bank amounts. Cash = payment_mode 'cash'; Bank = all others.
     */
    private function cashBankSplit(float $cashAmount, float $bankAmount): array
    {
        return [
            'total' => round($cashAmount + $bankAmount, 2),
            'cash' => round($cashAmount, 2),
            'bank' => round($bankAmount, 2),
        ];
    }

    /**
     * Display the admin dashboard with real KPIs, financial matrix, room inventory, and charts.
     */
    public function index(Request $request): View
    {
        $today = Carbon::today();

        // KPI 1: Today & All Registration
        $allStudentsCount = Student::query()->count();
        $registeredToday = Student::query()->whereDate('created_at', $today)->count();

        // KPI 2: Active / Total Students
        $activeStudents = Student::query()->where('status', Student::STATUS_ACTIVE)->count();
        $totalStudents = $allStudentsCount;

        // KPI 3: Today's & Total Expense
        $todayExpense = (float) Expense::query()->whereDate('expense_date', $today)->sum('amount');
        $totalExpenseAll = (float) Expense::query()->sum('amount');

        // KPI 4: Beds
        $totalBeds = Bed::query()->count();
        $occupiedBeds = Bed::query()->where('status', Bed::STATUS_OCCUPIED)->count();
        $availableBeds = $totalBeds - $occupiedBeds;
        $occupancyPercent = $totalBeds > 0 ? round(($occupiedBeds / $totalBeds) * 100) : 0;

        // Financial matrix: Opening (before today), Today's Collection, Today's Expenses, Total Amount (Total / Cash / Bank)
        $openingCash = $this->openingBalanceForMode($today, 'cash');
        $openingBank = $this->openingBalanceForMode($today, 'bank');
        $todayCollectionCash = $this->todayCollectionForMode($today, 'cash');
        $todayCollectionBank = $this->todayCollectionForMode($today, 'bank');
        $todayExpenseCash = $this->todayExpenseForMode($today, 'cash');
        $todayExpenseBank = $this->todayExpenseForMode($today, 'bank');

        $opening = $this->cashBankSplit($openingCash, $openingBank);
        $todayCollection = $this->cashBankSplit($todayCollectionCash, $todayCollectionBank);
        $todayExpensesRow = $this->cashBankSplit($todayExpenseCash, $todayExpenseBank);
        $totalAmount = [
            'total' => $opening['total'] + $todayCollection['total'] - $todayExpensesRow['total'],
            'cash' => $opening['cash'] + $todayCollection['cash'] - $todayExpensesRow['cash'],
            'bank' => $opening['bank'] + $todayCollection['bank'] - $todayExpensesRow['bank'],
        ];

        $collectionDetails = [
            ['label' => 'Opening Balance', 'total' => $opening['total'], 'cash' => $opening['cash'], 'bank' => $opening['bank']],
            ['label' => "Today's Collection", 'total' => $todayCollection['total'], 'cash' => $todayCollection['cash'], 'bank' => $todayCollection['bank']],
            ['label' => "Today's Expenses", 'total' => $todayExpensesRow['total'], 'cash' => $todayExpensesRow['cash'], 'bank' => $todayExpensesRow['bank']],
            ['label' => 'Total Amount', 'total' => $totalAmount['total'], 'cash' => $totalAmount['cash'], 'bank' => $totalAmount['bank']],
        ];

        // Room inventory: floors -> rooms -> beds (true = available, false = occupied)
        $floors = Floor::query()
            ->with(['rooms' => fn ($q) => $q->orderBy('name')->with(['beds' => fn ($q) => $q->orderBy('name')])])
            ->orderBy('sort_order')
            ->get()
            ->map(function (Floor $floor) {
                return [
                    'id' => $floor->id,
                    'name' => $floor->name,
                    'rooms' => $floor->rooms->map(function ($room) {
                        return [
                            'id' => $room->id,
                            'number' => $room->name,
                            'beds' => $room->beds->map(fn ($bed) => $bed->status === Bed::STATUS_AVAILABLE && ! $bed->student_id)->values()->all(),
                        ];
                    })->values()->all(),
                ];
            })
            ->all();

        $totalRooms = Floor::query()->withCount('rooms')->get()->sum('rooms_count');
        $occupancyRate = $totalRooms > 0 ? round(($activeStudents / $totalRooms) * 100) : 0;
        $dateFrom = now()->startOfMonth()->format('d/m/Y');
        $dateTo = now()->format('d/m/Y');

        // Collection MTD (month-to-date: 1st of month to today)
        $mtdRent = (float) Rent::query()
            ->whereDate('paid_at', '>=', now()->startOfMonth())
            ->whereDate('paid_at', '<=', $today)
            ->sum('amount');
        $mtdDeposit = (float) Deposit::query()
            ->whereDate('actual_deposit_date', '>=', now()->startOfMonth())
            ->whereDate('actual_deposit_date', '<=', $today)
            ->sum('amount');
        $collectionMtd = $mtdRent + $mtdDeposit;

        // Monthly trend (last 6 months) for chart
        $monthlyCollectionLabels = [];
        $monthlyCollectionData = [];
        $monthlyExpenseData = [];
        for ($i = 5; $i >= 0; $i--) {
            $month = now()->subMonths($i);
            $monthlyCollectionLabels[] = $month->format('M Y');
            $monthlyCollectionData[] = round(
                ((float) Rent::query()->whereYear('paid_at', $month->year)->whereMonth('paid_at', $month->month)->sum('amount') +
                (float) Deposit::query()->whereYear('actual_deposit_date', $month->year)->whereMonth('actual_deposit_date', $month->month)->sum('amount')) / 1000,
                1
            );
            $monthlyExpenseData[] = round(
                (float) Expense::query()->whereYear('expense_date', $month->year)->whereMonth('expense_date', $month->month)->sum('amount') / 1000,
                1
            );
        }

        return view('admin.dashboard', [
            'today' => now()->format('l, d F Y H:i'),
            'registeredToday' => $registeredToday,
            'allStudentsCount' => $allStudentsCount,
            'activeStudents' => $activeStudents,
            'totalStudents' => $totalStudents,
            'todayExpense' => $todayExpense,
            'totalExpenseAll' => $totalExpenseAll,
            'totalBeds' => $totalBeds,
            'occupiedBeds' => $occupiedBeds,
            'availableBeds' => $availableBeds,
            'occupancyPercent' => $occupancyPercent,
            'totalRooms' => $totalRooms,
            'occupancyRate' => $occupancyRate,
            'collectionDetails' => $collectionDetails,
            'floors' => $floors,
            'dateFrom' => $dateFrom,
            'dateTo' => $dateTo,
            'collectionMtd' => $collectionMtd,
            'monthlyCollectionLabels' => $monthlyCollectionLabels,
            'monthlyCollectionData' => $monthlyCollectionData,
            'monthlyExpenseData' => $monthlyExpenseData,
        ]);
    }

    /**
     * Opening balance for a payment mode (all collection - all expense before today).
     * Includes: Drawings, Bank Deposits, and Bank Withdrawals.
     */
    private function openingBalanceForMode(Carbon $today, string $mode): float
    {
        $cashOnly = $mode === 'cash';
        
        // Collection (Rent + Deposit)
        $rentQuery = Rent::query()->where('paid_at', '<', $today->copy()->startOfDay());
        $depositQuery = Deposit::query()->where('actual_deposit_date', '<', $today);
        
        // Outgoing (Expenses)
        $expenseQuery = Expense::query()->where('expense_date', '<', $today);
        
        if ($cashOnly) {
            $rentQuery->where('payment_mode', 'cash');
            $depositQuery->where('payment_mode', 'cash');
            $expenseQuery->where('payment_mode', 'cash');
        } else {
            $rentQuery->where('payment_mode', '!=', 'cash');
            $depositQuery->where('payment_mode', '!=', 'cash');
            $expenseQuery->where('payment_mode', '!=', 'cash');
        }
        
        $collection = (float) $rentQuery->sum('amount') + (float) $depositQuery->sum('amount');
        $expense = (float) $expenseQuery->sum('amount');
        
        // Drawing adjustments (all drawings affect CASH only regardless of payment_mode)
        if ($cashOnly) {
            $drawingQuery = Drawing::query()->where('drawing_date', '<', $today);
            // Credits increase cash (funds in), Debits decrease cash (funds out - cash taken home)
            $drawingCredit = (float) $drawingQuery->clone()->where('type', Drawing::TYPE_CREDIT)->sum('amount');
            $drawingDebit = (float) $drawingQuery->clone()->where('type', Drawing::TYPE_DEBIT)->sum('amount');
            $collection += $drawingCredit;
            $expense += $drawingDebit;
        }
        
        // Bank Entry adjustments
        $bankEntryQuery = BankEntry::query()->where('entry_date', '<', $today);
        if ($cashOnly) {
            // Cash: Withdrawals increase cash, Deposits decrease cash
            $withdrawals = (float) $bankEntryQuery->clone()->where('type', BankEntry::TYPE_WITHDRAW_FROM_BANK)->sum('amount');
            $deposits = (float) $bankEntryQuery->clone()->where('type', BankEntry::TYPE_DEPOSIT_TO_BANK)->sum('amount');
            $collection += $withdrawals; // Cash increases
            $expense += $deposits; // Cash decreases
        } else {
            // Bank: Deposits increase bank, Withdrawals decrease bank
            $deposits = (float) $bankEntryQuery->clone()->where('type', BankEntry::TYPE_DEPOSIT_TO_BANK)->sum('amount');
            $withdrawals = (float) $bankEntryQuery->clone()->where('type', BankEntry::TYPE_WITHDRAW_FROM_BANK)->sum('amount');
            $collection += $deposits; // Bank increases
            $expense += $withdrawals; // Bank decreases
        }
        
        return $collection - $expense;
    }

    private function todayCollectionForMode(Carbon $today, string $mode): float
    {
        $cashOnly = $mode === 'cash';
        
        // Collection (Rent + Deposit)
        $rentQuery = Rent::query()->whereDate('paid_at', $today);
        $depositQuery = Deposit::query()->whereDate('actual_deposit_date', $today);
        
        if ($cashOnly) {
            $rentQuery->where('payment_mode', 'cash');
            $depositQuery->where('payment_mode', 'cash');
        } else {
            $rentQuery->where('payment_mode', '!=', 'cash');
            $depositQuery->where('payment_mode', '!=', 'cash');
        }
        
        $collection = (float) $rentQuery->sum('amount') + (float) $depositQuery->sum('amount');
        
        // Drawing credits (funds in) - affects CASH only
        if ($cashOnly) {
            $drawingCredit = (float) Drawing::query()
                ->whereDate('drawing_date', $today)
                ->where('type', Drawing::TYPE_CREDIT)
                ->sum('amount');
            $collection += $drawingCredit;
        }
        
        // Bank Entry adjustments
        $bankEntryQuery = BankEntry::query()->whereDate('entry_date', $today);
        if ($cashOnly) {
            // Cash: Withdrawals increase cash
            $withdrawals = (float) $bankEntryQuery->where('type', BankEntry::TYPE_WITHDRAW_FROM_BANK)->sum('amount');
            $collection += $withdrawals;
        } else {
            // Bank: Deposits increase bank
            $deposits = (float) $bankEntryQuery->where('type', BankEntry::TYPE_DEPOSIT_TO_BANK)->sum('amount');
            $collection += $deposits;
        }
        
        return $collection;
    }

    private function todayExpenseForMode(Carbon $today, string $mode): float
    {
        $cashOnly = $mode === 'cash';
        
        // Regular Expenses
        $query = Expense::query()->whereDate('expense_date', $today);
        if ($cashOnly) {
            $query->where('payment_mode', 'cash');
        } else {
            $query->where('payment_mode', '!=', 'cash');
        }
        
        $expense = (float) $query->sum('amount');
        
        // Drawing debits (cash taken home) - affects CASH only
        if ($cashOnly) {
            $drawingDebit = (float) Drawing::query()
                ->whereDate('drawing_date', $today)
                ->where('type', Drawing::TYPE_DEBIT)
                ->sum('amount');
            $expense += $drawingDebit;
        }
        
        // Bank Entry adjustments
        $bankEntryQuery = BankEntry::query()->whereDate('entry_date', $today);
        if ($cashOnly) {
            // Cash: Deposits decrease cash
            $deposits = (float) $bankEntryQuery->where('type', BankEntry::TYPE_DEPOSIT_TO_BANK)->sum('amount');
            $expense += $deposits;
        } else {
            // Bank: Withdrawals decrease bank
            $withdrawals = (float) $bankEntryQuery->where('type', BankEntry::TYPE_WITHDRAW_FROM_BANK)->sum('amount');
            $expense += $withdrawals;
        }
        
        return $expense;
    }

    /**
     * JSON endpoint for real-time dashboard refresh (AJAX).
     */
    public function stats(Request $request): JsonResponse
    {
        $today = Carbon::today();

        $registeredToday = Student::query()->whereDate('created_at', $today)->count();
        $allStudentsCount = Student::query()->count();
        $activeStudents = Student::query()->where('status', Student::STATUS_ACTIVE)->count();
        $todayExpense = (float) Expense::query()->whereDate('expense_date', $today)->sum('amount');
        $totalExpenseAll = (float) Expense::query()->sum('amount');

        $totalBeds = Bed::query()->count();
        $occupiedBeds = Bed::query()->where('status', Bed::STATUS_OCCUPIED)->count();
        $availableBeds = $totalBeds - $occupiedBeds;
        $occupancyPercent = $totalBeds > 0 ? round(($occupiedBeds / $totalBeds) * 100) : 0;

        $openingCash = $this->openingBalanceForMode($today, 'cash');
        $openingBank = $this->openingBalanceForMode($today, 'bank');
        $todayCollectionCash = $this->todayCollectionForMode($today, 'cash');
        $todayCollectionBank = $this->todayCollectionForMode($today, 'bank');
        $todayExpenseCash = $this->todayExpenseForMode($today, 'cash');
        $todayExpenseBank = $this->todayExpenseForMode($today, 'bank');

        $opening = $this->cashBankSplit($openingCash, $openingBank);
        $todayCollection = $this->cashBankSplit($todayCollectionCash, $todayCollectionBank);
        $todayExpensesRow = $this->cashBankSplit($todayExpenseCash, $todayExpenseBank);
        $totalAmount = [
            'total' => $opening['total'] + $todayCollection['total'] - $todayExpensesRow['total'],
            'cash' => $opening['cash'] + $todayCollection['cash'] - $todayExpensesRow['cash'],
            'bank' => $opening['bank'] + $todayCollection['bank'] - $todayExpensesRow['bank'],
        ];

        $mtdRent = (float) Rent::query()
            ->whereDate('paid_at', '>=', now()->startOfMonth())
            ->whereDate('paid_at', '<=', $today)
            ->sum('amount');
        $mtdDeposit = (float) Deposit::query()
            ->whereDate('actual_deposit_date', '>=', now()->startOfMonth())
            ->whereDate('actual_deposit_date', '<=', $today)
            ->sum('amount');
        $collectionMtd = $mtdRent + $mtdDeposit;

        $totalRooms = Floor::query()->withCount('rooms')->get()->sum('rooms_count');
        $occupancyRate = $totalRooms > 0 ? round(($activeStudents / $totalRooms) * 100) : 0;

        return response()->json([
            'registered_today' => $registeredToday,
            'all_students_count' => $allStudentsCount,
            'active_students' => $activeStudents,
            'total_students' => $allStudentsCount,
            'today_expense' => $todayExpense,
            'total_expense_all' => $totalExpenseAll,
            'total_beds' => $totalBeds,
            'occupied_beds' => $occupiedBeds,
            'available_beds' => $availableBeds,
            'occupancy_percent' => $occupancyPercent,
            'total_rooms' => $totalRooms,
            'occupancy_rate' => $occupancyRate,
            'collection_details' => [
                ['label' => 'Opening Balance', 'total' => $opening['total'], 'cash' => $opening['cash'], 'bank' => $opening['bank']],
                ['label' => "Today's Collection", 'total' => $todayCollection['total'], 'cash' => $todayCollection['cash'], 'bank' => $todayCollection['bank']],
                ['label' => "Today's Expenses", 'total' => $todayExpensesRow['total'], 'cash' => $todayExpensesRow['cash'], 'bank' => $todayExpensesRow['bank']],
                ['label' => 'Total Amount', 'total' => $totalAmount['total'], 'cash' => $totalAmount['cash'], 'bank' => $totalAmount['bank']],
            ],
            'collection_mtd' => $collectionMtd,
        ]);
    }
}
