<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Drawing;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class DrawingController extends Controller
{
    /**
     * Drawing list: credit (funds in) / debit (funds out). Filters + total.
     */
    public function index(Request $request): View
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');
        $type = $request->query('type');
        $paymentMode = $request->query('payment_mode');

        $query = Drawing::query()
            ->with('performedBy')
            ->orderByDesc('drawing_date')
            ->orderByDesc('id');

        if ($fromDate) {
            $query->whereDate('drawing_date', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('drawing_date', '<=', $toDate);
        }
        if ($type !== null && $type !== '') {
            $query->where('type', $type);
        }
        if ($paymentMode !== null && $paymentMode !== '') {
            $query->where('payment_mode', $paymentMode);
        }

        $drawings = $query->get();
        $totalDebit = $drawings->where('type', Drawing::TYPE_DEBIT)->sum('amount');
        $totalCredit = $drawings->where('type', Drawing::TYPE_CREDIT)->sum('amount');
        $total = $totalCredit - $totalDebit;

        $paymentModes = config('student_registration.payment_modes', []);

        return view('admin.drawings.index', [
            'drawings' => $drawings,
            'totalDebit' => $totalDebit,
            'totalCredit' => $totalCredit,
            'total' => $total,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'type' => $type,
            'paymentMode' => $paymentMode,
            'paymentModes' => $paymentModes,
        ]);
    }

    /**
     * Store a drawing (credit or debit). Captures current admin user for audit.
     */
    public function store(Request $request): RedirectResponse
    {
        $paymentModes = array_keys(config('student_registration.payment_modes', []));
        $validated = $request->validate([
            'drawing_date' => ['required', 'date'],
            'amount' => ['required', 'numeric', 'min:0.01'],
            'type' => ['required', Rule::in([Drawing::TYPE_CREDIT, Drawing::TYPE_DEBIT])],
            'payment_mode' => ['required', Rule::in($paymentModes)],
            'transaction_id' => ['nullable', 'string', 'max:255'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        Drawing::query()->create([
            'drawing_date' => $validated['drawing_date'],
            'amount' => $validated['amount'],
            'type' => $validated['type'],
            'payment_mode' => $validated['payment_mode'],
            'transaction_id' => $validated['transaction_id'] ?? null,
            'notes' => $validated['notes'] ?? null,
            'performed_by_id' => auth()->guard('admin')->id(),
        ]);

        $label = $validated['type'] === Drawing::TYPE_CREDIT ? 'Credit (funds in)' : 'Debit (funds out)';
        return redirect()
            ->route('admin.drawings.index')
            ->with('success', $label . ' recorded successfully.');
    }
}
