<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Enquiry;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class EnquiryController extends Controller
{
    /**
     * Enquiry list with filters (date range, status). Optional AJAX for table body.
     */
    public function index(Request $request): View|JsonResponse
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');
        $status = $request->query('status');

        $query = Enquiry::query()->orderByDesc('created_at');

        if ($fromDate) {
            $query->whereDate('created_at', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('created_at', '<=', $toDate);
        }
        if ($status !== null && $status !== '') {
            $query->where('status', $status);
        } else {
            $query->where('status', '!=', Enquiry::STATUS_DELETED);
        }

        $enquiries = $query->get();

        if ($request->wantsJson() || $request->ajax()) {
            return response()->json([
                'html' => view('admin.enquiries._table_body', [
                    'enquiries' => $enquiries,
                ])->render(),
            ]);
        }

        return view('admin.enquiries.index', [
            'enquiries' => $enquiries,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'status' => $status,
        ]);
    }

    /**
     * Store a new enquiry (from Add Enquiry form/modal).
     */
    public function store(Request $request): RedirectResponse|JsonResponse
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'mobile' => ['required', 'string', 'max:20'],
            'class' => ['nullable', 'string', 'max:255'],
            'address' => ['nullable', 'string', 'max:500'],
            'message' => ['nullable', 'string', 'max:2000'],
        ]);

        Enquiry::query()->create([
            'name' => $validated['name'],
            'mobile' => $validated['mobile'],
            'class' => $validated['class'] ?? null,
            'address' => $validated['address'] ?? null,
            'message' => $validated['message'] ?? null,
            'status' => Enquiry::STATUS_PENDING,
        ]);

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Enquiry added successfully.']);
        }

        return redirect()
            ->route('admin.enquiries.index')
            ->with('success', 'Enquiry added successfully.');
    }

    /**
     * Mark enquiry as checked.
     */
    public function check(Enquiry $enquiry): RedirectResponse
    {
        $enquiry->update(['status' => Enquiry::STATUS_CHECKED]);

        return redirect()
            ->route('admin.enquiries.index', request()->only(['from_date', 'to_date', 'status']))
            ->with('success', 'Enquiry marked as checked.');
    }

    /**
     * Add remark and set status to checked.
     */
    public function storeRemark(Request $request, Enquiry $enquiry): RedirectResponse
    {
        $validated = $request->validate([
            'remark' => ['required', 'string', 'max:2000'],
        ]);

        $enquiry->update([
            'remark' => $validated['remark'],
            'status' => Enquiry::STATUS_CHECKED,
        ]);

        return redirect()
            ->route('admin.enquiries.index', request()->only(['from_date', 'to_date', 'status']))
            ->with('success', 'Remark added and enquiry marked as checked.');
    }

    /**
     * Soft delete: set status to 'delete'.
     */
    public function destroy(Enquiry $enquiry): RedirectResponse
    {
        $enquiry->update(['status' => Enquiry::STATUS_DELETED]);

        return redirect()
            ->route('admin.enquiries.index', request()->only(['from_date', 'to_date', 'status']))
            ->with('success', 'Enquiry removed from list.');
    }
}
