<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Expense;
use App\Models\ExpenseHead;
use App\Models\Rent;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class ExpenseController extends Controller
{
    /**
     * Expenses Details: list with filters (date range, head, payment mode), total, add form.
     */
    public function index(Request $request): View
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');
        $headId = $request->query('head_id');
        $paymentMode = $request->query('payment_mode');

        $query = Expense::query()
            ->with('expenseHead')
            ->orderByDesc('expense_date')
            ->orderByDesc('id');

        if ($fromDate) {
            $query->whereDate('expense_date', '>=', $fromDate);
        }
        if ($toDate) {
            $query->whereDate('expense_date', '<=', $toDate);
        }
        if ($headId !== null && $headId !== '') {
            $query->where('expense_head_id', $headId);
        }
        if ($paymentMode !== null && $paymentMode !== '') {
            $query->where('payment_mode', $paymentMode);
        }

        $expenses = $query->get();
        $total = $expenses->sum('amount');

        $heads = ExpenseHead::query()->orderBy('name')->get(['id', 'name']);
        $paymentModes = config('student_registration.payment_modes', []);

        return view('admin.expenses.index', [
            'expenses' => $expenses,
            'total' => $total,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'headId' => $headId,
            'paymentMode' => $paymentMode,
            'heads' => $heads,
            'paymentModes' => $paymentModes,
        ]);
    }

    /**
     * Store a new expense (with optional bill file upload). Validates balance per payment mode.
     */
    public function store(Request $request): RedirectResponse
    {
        $paymentModes = array_keys(config('student_registration.payment_modes', []));
        $validated = $request->validate([
            'expense_head_id' => ['required', 'integer', 'exists:expense_heads,id'],
            'expense_date' => ['required', 'date'],
            'item_name' => ['required', 'string', 'max:255'],
            'amount' => ['required', 'numeric', 'min:0'],
            'payment_mode' => ['required', Rule::in($paymentModes)],
            'transaction_id' => ['nullable', 'string', 'max:255'],
            'bill_file' => ['nullable', 'file', 'max:5120', 'mimes:jpg,jpeg,png,pdf'],
        ], [], ['bill_file' => 'bill/receipt file']);

        $amount = (float) $validated['amount'];
        $mode = $validated['payment_mode'];

        $income = (float) Rent::query()->where('payment_mode', $mode)->sum('amount')
            + (float) Deposit::query()->where('payment_mode', $mode)->sum('amount');
        $alreadyExpensed = (float) Expense::query()->where('payment_mode', $mode)->sum('amount');
        $balance = $income - $alreadyExpensed;

        if ($amount > $balance) {
            return back()
                ->withInput()
                ->withErrors(['amount' => 'Amount is not available. Balance for ' . (config('student_registration.payment_modes.' . $mode, $mode)) . ': ₹' . number_format($balance, 2) . '.']);
        }

        $filePath = null;
        if ($request->hasFile('bill_file')) {
            $file = $request->file('bill_file');
            $filePath = $file->store('expenses', 'public');
        }

        Expense::query()->create([
            'expense_head_id' => $validated['expense_head_id'],
            'expense_date' => $validated['expense_date'],
            'item_name' => $validated['item_name'],
            'amount' => $amount,
            'payment_mode' => $mode,
            'transaction_id' => $validated['transaction_id'] ?? null,
            'file_path' => $filePath,
        ]);

        return redirect()
            ->route('admin.expenses.index')
            ->with('success', 'Expense recorded successfully.');
    }

    /**
     * Return current balance for a payment mode (for AJAX balance check).
     */
    public function balance(Request $request)
    {
        $mode = $request->query('payment_mode');
        if (! $mode) {
            return response()->json(['balance' => 0]);
        }
        $income = (float) Rent::query()->where('payment_mode', $mode)->sum('amount')
            + (float) Deposit::query()->where('payment_mode', $mode)->sum('amount');
        $expensed = (float) Expense::query()->where('payment_mode', $mode)->sum('amount');
        return response()->json(['balance' => $income - $expensed]);
    }
}
