<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Floor;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class FloorController extends Controller
{
    public function index()
    {
        $floors = Floor::query()->withCount('rooms')->orderBy('sort_order')->orderBy('name')->get();
        return view('admin.master.floors.index', compact('floors'));
    }

    public function show(Floor $floor)
    {
        $floor->loadCount('rooms');
        $floor->load('rooms.beds');
        return view('admin.master.floors.show', compact('floor'));
    }

    public function create()
    {
        return view('admin.master.floors.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('floors', 'name')],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'capacity' => ['nullable', 'integer', 'min:0'],
        ]);

        $validated['sort_order'] = $validated['sort_order'] ?? 0;
        $validated['capacity'] = $validated['capacity'] ?? 0;
        Floor::query()->create($validated);

        return redirect()
            ->route('admin.floors.index')
            ->with('success', 'Floor created successfully.');
    }

    public function edit(Floor $floor)
    {
        return view('admin.master.floors.edit', compact('floor'));
    }

    public function update(Request $request, Floor $floor)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('floors', 'name')->ignore($floor->id)],
            'sort_order' => ['nullable', 'integer', 'min:0'],
            'capacity' => ['nullable', 'integer', 'min:0'],
        ]);
        $floor->update([
            'name' => $validated['name'],
            'sort_order' => $validated['sort_order'] ?? 0,
            'capacity' => $validated['capacity'] ?? 0,
        ]);
        return redirect()->route('admin.floors.index')->with('success', 'Floor updated successfully.');
    }

    public function destroy(Floor $floor)
    {
        $floor->delete();
        return redirect()
            ->route('admin.floors.index')
            ->with('success', 'Floor deleted successfully.');
    }
}
