<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Leave;
use App\Models\Student;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class LeaveController extends Controller
{
    /**
     * Leave list with filters: status (pending, approved, returned), date range.
     */
    public function index(Request $request): View
    {
        $statusFilter = $request->query('status', 'all'); // all, pending, approved, returned
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');

        $query = Leave::query()
            ->with(['student.documents'])
            ->orderByDesc('from_date')
            ->orderByDesc('id');

        if ($statusFilter === 'pending') {
            $query->where('status', Leave::STATUS_PENDING);
        } elseif ($statusFilter === 'approved') {
            $query->where('status', Leave::STATUS_APPROVED)->whereNull('return_date');
        } elseif ($statusFilter === 'returned') {
            $query->where('status', Leave::STATUS_APPROVED)->whereNotNull('return_date');
        } elseif ($statusFilter === 'cancelled') {
            $query->where('status', Leave::STATUS_CANCELLED);
        }

        if ($fromDate && $toDate) {
            $query->where(function ($q) use ($fromDate, $toDate) {
                $q->whereBetween('from_date', [$fromDate, $toDate])
                    ->orWhereBetween('to_date', [$fromDate, $toDate])
                    ->orWhereBetween('return_date', [$fromDate, $toDate]);
            });
        }

        $leaves = $query->get();

        $counts = [
            'pending' => Leave::query()->where('status', Leave::STATUS_PENDING)->count(),
            'approved' => Leave::query()->where('status', Leave::STATUS_APPROVED)->whereNull('return_date')->count(),
            'returned' => Leave::query()->where('status', Leave::STATUS_APPROVED)->whereNotNull('return_date')->count(),
        ];

        return view('admin.leaves.index', [
            'leaves' => $leaves,
            'statusFilter' => $statusFilter,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'counts' => $counts,
        ]);
    }

    /**
     * Add Student Leave form.
     */
    public function create(): View
    {
        $students = Student::query()
            ->with('bed')
            ->where('status', Student::STATUS_ACTIVE)
            ->where('form_status', Student::FORM_STATUS_ACTIVE)
            ->orderBy('first_name')
            ->orderBy('last_name')
            ->get(['id', 'first_name', 'middle_name', 'last_name', 'student_mobile']);

        return view('admin.leaves.create', ['students' => $students]);
    }

    /**
     * Store new leave (admin entry).
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'student_id' => ['required', 'integer', 'exists:students,id'],
            'from_date' => ['required', 'date'],
            'to_date' => ['required', 'date', 'after_or_equal:from_date'],
            'reason' => ['required', 'string', 'max:2000'],
        ]);

        Leave::query()->create([
            'student_id' => $validated['student_id'],
            'from_date' => $validated['from_date'],
            'to_date' => $validated['to_date'],
            'reason' => $validated['reason'],
            'status' => Leave::STATUS_PENDING,
        ]);

        return redirect()
            ->route('admin.leaves.index')
            ->with('success', 'Leave request added. It is pending approval.');
    }

    /**
     * Approve a pending leave.
     */
    public function approve(Leave $leave): RedirectResponse
    {
        if ($leave->status !== Leave::STATUS_PENDING) {
            return redirect()->route('admin.leaves.index')->with('error', 'Leave is not pending.');
        }

        $leave->update(['status' => Leave::STATUS_APPROVED]);

        return redirect()
            ->route('admin.leaves.index')
            ->with('success', 'Leave approved for ' . $leave->student->full_name . '.');
    }

    /**
     * Cancel (reject) a pending leave.
     */
    public function cancel(Leave $leave): RedirectResponse
    {
        if ($leave->status !== Leave::STATUS_PENDING) {
            return redirect()->route('admin.leaves.index')->with('error', 'Only pending leaves can be cancelled.');
        }

        $leave->update(['status' => Leave::STATUS_CANCELLED]);

        return redirect()
            ->route('admin.leaves.index')
            ->with('success', 'Leave request cancelled.');
    }

    /**
     * Record student return (set return date).
     */
    public function storeReturn(Request $request, Leave $leave): RedirectResponse
    {
        $validated = $request->validate([
            'return_date' => ['required', 'date', 'after_or_equal:from_date'],
        ], [], ['return_date' => 'return date']);

        if ($leave->status !== Leave::STATUS_APPROVED) {
            return redirect()->route('admin.leaves.index')->with('error', 'Only approved leaves can be marked returned.');
        }

        $leave->update(['return_date' => $validated['return_date']]);

        return redirect()
            ->route('admin.leaves.index')
            ->with('success', 'Return date recorded for ' . $leave->student->full_name . '.');
    }
}
