<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\NotificationService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class NotificationController extends Controller
{
    protected NotificationService $notificationService;

    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
    }

    /**
     * Display all notifications page (no sync on load - use Refresh button or cron)
     */
    public function index(Request $request): View
    {
        $filter = $request->get('filter', 'all');
        $counts = $this->notificationService->getCounts();

        return view('admin.notifications.index', [
            'unreadCount' => $counts['unread'],
            'criticalCount' => $counts['critical'],
            'totalCount' => $counts['total'],
            'filter' => $filter,
        ]);
    }

    /**
     * Get notifications data for DataTables (server-side processing)
     */
    public function getData(Request $request): JsonResponse
    {
        $draw = (int) $request->input('draw', 1);
        $start = (int) $request->input('start', 0);
        $length = (int) $request->input('length', 25);
        $search = $request->input('search');
        $searchValue = is_array($search) && isset($search['value']) ? (string) $search['value'] : (string) $request->input('search.value', '');
        $filter = $request->input('filter', 'all');

        $query = \App\Models\AdminNotification::query()
            ->orderBy('created_at', 'desc');

        // Apply filter
        if ($filter === 'unread') {
            $query->unread();
        } elseif ($filter === 'critical') {
            $query->where('priority', 'critical');
        }

        // Total records
        $totalRecords = (clone $query)->count();

        // Apply search
        if ($searchValue) {
            $query->where(function ($q) use ($searchValue) {
                $q->where('title', 'like', "%{$searchValue}%")
                    ->orWhere('message', 'like', "%{$searchValue}%")
                    ->orWhere('type', 'like', "%{$searchValue}%");
            });
        }

        // Filtered records count
        $filteredRecords = (clone $query)->count();

        // Get paginated data
        $notifications = $query->skip($start)->take($length)->get();

        $typeLabels = [
            'agreement_expiring' => 'Agreement expiring',
            'rent_due' => 'Rent due',
            'rent_overdue' => 'Rent overdue',
            'deposit_low' => 'Low deposit',
            'student_pending' => 'Registration pending',
        ];

        // Format data for DataTables (meta as object for JS)
        $data = $notifications->map(function ($notification) use ($typeLabels) {
            return [
                'id' => $notification->id,
                'type' => $notification->type,
                'type_label' => $typeLabels[$notification->type] ?? ucfirst(str_replace('_', ' ', $notification->type)),
                'priority' => $notification->priority,
                'title' => $notification->title,
                'message' => $notification->message,
                'meta' => $notification->meta ?? [],
                'action_url' => $notification->action_url ?? '',
                'is_read' => (bool) $notification->is_read,
                'time_ago' => $notification->time_ago,
                'created_at' => $notification->created_at->format('Y-m-d H:i:s'),
            ];
        });

        return response()->json([
            'draw' => $draw,
            'recordsTotal' => $totalRecords,
            'recordsFiltered' => $filteredRecords,
            'data' => $data->values()->all(),
        ]);
    }

    /**
     * Get category counts for notification bar (AJAX) - fast, no full list
     */
    public function dropdown(Request $request): JsonResponse
    {
        $counts = $this->notificationService->getCounts();
        $categories = $this->notificationService->getCategoryCounts();

        return response()->json([
            'categories' => $categories,
            'unread_count' => $counts['unread'],
            'critical_count' => $counts['critical'],
            'total_count' => $counts['total'],
        ]);
    }

    /**
     * Mark notification as read
     */
    public function markAsRead(int $id): JsonResponse
    {
        $this->notificationService->markAsRead($id);

        return response()->json([
            'success' => true,
            'unread_count' => $this->notificationService->getUnreadCount(),
        ]);
    }

    /**
     * Mark all notifications as read
     */
    public function markAllAsRead(): JsonResponse
    {
        $this->notificationService->markAllAsRead();

        return response()->json([
            'success' => true,
            'unread_count' => 0,
        ]);
    }

    /**
     * Sync notifications (can be called via cron or manually)
     */
    public function sync(): JsonResponse
    {
        $this->notificationService->syncNotifications();
        $counts = $this->notificationService->getCounts();

        return response()->json([
            'success' => true,
            'unread_count' => $counts['unread'],
            'critical_count' => $counts['critical'],
            'total_count' => $counts['total'],
        ]);
    }
}
