<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class ProfileController extends Controller
{
    /**
     * Show Account Settings (Account + Security tabs). Optional ?tab=security.
     */
    public function edit(Request $request): View
    {
        $user = Auth::guard('admin')->user();
        $activeTab = $request->query('tab', 'account');

        return view('admin.profile.edit', [
            'user' => $user,
            'activeTab' => $activeTab === 'security' ? 'security' : 'account',
        ]);
    }

    /**
     * Update admin profile (name, email, optional avatar).
     */
    public function update(Request $request): RedirectResponse
    {
        $user = Auth::guard('admin')->user();

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users', 'email')->ignore($user->id),
            ],
            'phone' => ['nullable', 'string', 'max:20'],
            'date_of_birth' => ['nullable', 'date'],
            'organization' => ['nullable', 'string', 'max:255'],
            'designation' => ['nullable', 'string', 'max:255'],
            'avatar' => ['nullable', 'image', 'mimes:jpg,jpeg,png,gif', 'max:800'],
        ]);

        $data = [
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'] ?? null,
            'date_of_birth' => $validated['date_of_birth'] ?? null,
            'organization' => $validated['organization'] ?? null,
            'designation' => $validated['designation'] ?? null,
        ];

        if ($request->hasFile('avatar')) {
            if ($user->avatar) {
                Storage::disk('public')->delete($user->avatar);
            }
            $path = $request->file('avatar')->store('avatars', 'public');
            $data['avatar'] = $path;
        }

        if ($request->boolean('avatar_remove')) {
            if ($user->avatar) {
                Storage::disk('public')->delete($user->avatar);
            }
            $data['avatar'] = null;
        }

        $user->update($data);

        return redirect()
            ->route('admin.profile.edit', ['tab' => 'account'])
            ->with('success', 'Profile updated successfully.');
    }

    /**
     * Show change password (redirects to Account Settings with Security tab).
     */
    public function showChangePasswordForm(): RedirectResponse
    {
        return redirect()->route('admin.profile.edit', ['tab' => 'security']);
    }

    /**
     * Update password (current password required).
     */
    public function updatePassword(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'current_password' => ['required', 'string'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);

        $user = Auth::guard('admin')->user();
        if (!Hash::check($validated['current_password'], $user->password)) {
            return back()->withErrors(['current_password' => 'The current password is incorrect.'])->withInput();
        }

        $user->update([
            'password' => Hash::make($validated['password']),
        ]);

        return redirect()
            ->route('admin.profile.edit', ['tab' => 'security'])
            ->with('success', 'Password changed successfully.');
    }
}
