<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Rent;
use App\Models\Student;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Illuminate\View\View;

class RentController extends Controller
{
    /**
     * Rent Management: Rent Paid (current month) + Rent Pending (collection list).
     */
    public function index(Request $request): View
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');
        $paymentMode = $request->query('payment_mode');
        $currentMonthStart = Carbon::now()->startOfMonth();
        $currentMonthEnd = Carbon::now()->endOfMonth();

        // Rent Paid: transactions where rent_date/period falls within current month (or filter range)
        $rentPaidQuery = Rent::query()
            ->with(['student.bed', 'student.documents'])
            ->where(function ($q) use ($fromDate, $toDate, $currentMonthStart, $currentMonthEnd) {
                if ($fromDate && $toDate) {
                    $q->whereBetween('period_from', [$fromDate, $toDate])
                        ->orWhereBetween('paid_at', [$fromDate, $toDate]);
                } else {
                    $q->whereBetween('period_from', [$currentMonthStart, $currentMonthEnd])
                        ->orWhereBetween('paid_at', [$currentMonthStart, $currentMonthEnd]);
                }
            });
        if ($paymentMode) {
            $rentPaidQuery->where('payment_mode', $paymentMode);
        }
        $rentPaid = $rentPaidQuery->orderByDesc('paid_at')->orderByDesc('id')->get();

        // Rent Pending: active residents with no rent record for current month (collection list)
        $rentPendingQuery = Student::query()
            ->with(['bed', 'documents'])
            ->where('status', Student::STATUS_ACTIVE)
            ->where('form_status', Student::FORM_STATUS_ACTIVE)
            ->whereDoesntHave('rents', function ($q) use ($currentMonthStart, $currentMonthEnd) {
                $q->whereBetween('period_from', [$currentMonthStart, $currentMonthEnd]);
            });
        $rentPending = $rentPendingQuery->orderBy('first_name')->get();

        $paymentModes = config('student_registration.payment_modes', []);
        $stats = [
            'rent_paid_count' => $rentPaid->count(),
            'rent_paid_total' => $rentPaid->sum('amount'),
            'rent_pending_count' => $rentPending->count(),
        ];

        return view('admin.rents.index', [
            'rentPaid' => $rentPaid,
            'rentPending' => $rentPending,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'paymentMode' => $paymentMode,
            'paymentModes' => $paymentModes,
            'stats' => $stats,
        ]);
    }

    /**
     * Add Rent form: multi-entry, optional student_id; balance/advance from bed rent.
     */
    public function create(Request $request): View
    {
        $paymentModes = config('student_registration.payment_modes', []);
        $students = Student::query()
            ->with('bed')
            ->where('status', Student::STATUS_ACTIVE)
            ->where('form_status', Student::FORM_STATUS_ACTIVE)
            ->orderBy('first_name')
            ->orderBy('last_name')
            ->get(['id', 'first_name', 'middle_name', 'last_name', 'student_mobile', 'bed_id', 'reward_points']);

        $selectedStudent = null;
        if ($request->filled('student_id')) {
            $id = (int) $request->query('student_id');
            $selectedStudent = $students->firstWhere('id', $id);
        }

        $minRewardForRedeem = (int) config('student_registration.reward_points_min_for_redemption', 150);

        return view('admin.rents.create', [
            'paymentModes' => $paymentModes,
            'students' => $students,
            'selectedStudent' => $selectedStudent,
            'minRewardForRedeem' => $minRewardForRedeem,
        ]);
    }

    /**
     * Store one or more rent entries (multi-month).
     */
    public function store(Request $request): RedirectResponse
    {
        $paymentModes = array_keys(config('student_registration.payment_modes', []));
        $validated = $request->validate([
            'student_id' => ['required', 'integer', 'exists:students,id'],
            'entries' => ['required', 'array', 'min:1'],
            'entries.*.period_from' => ['required', 'date'],
            'entries.*.period_to' => ['required', 'date', 'after_or_equal:entries.*.period_from'],
            'entries.*.due_date' => ['required', 'date'],
            'entries.*.amount' => ['required', 'numeric', 'min:0'],
            'entries.*.reward_discount' => ['nullable', 'numeric', 'min:0'],
            'payment_mode' => ['required', Rule::in($paymentModes)],
            'transaction_id' => ['nullable', 'string', 'max:255'],
            'cheque_number' => ['nullable', 'string', 'max:100'],
        ]);

        $student = Student::query()->with('bed')->findOrFail($validated['student_id']);
        $rentFeesPerMonth = $student->bed ? (float) $student->bed->rent_amount : 0;
        if ($rentFeesPerMonth <= 0) {
            $rentFeesPerMonth = 0; // allow manual amount if bed has no rent set
        }

        $lastBalance = 0;
        $lastAdvance = 0;
        $latestRent = $student->rents()->orderByDesc('paid_at')->first();
        if ($latestRent) {
            $lastBalance = (float) ($latestRent->balance_amount ?? 0);
            $lastAdvance = (float) ($latestRent->advance_amount ?? 0);
        }

        DB::transaction(function () use ($student, $validated, $rentFeesPerMonth, &$lastBalance, &$lastAdvance) {
            $baseSeq = Rent::query()->whereMonth('paid_at', now()->month)->whereYear('paid_at', now()->year)->count();
            foreach ($validated['entries'] as $entry) {
                $amount = (float) $entry['amount'];
                $discount = (float) ($entry['reward_discount'] ?? 0);
                $effectiveFees = max(0, $rentFeesPerMonth - $discount);
                if ($effectiveFees <= 0 && $rentFeesPerMonth <= 0) {
                    $effectiveFees = $amount; // no bed rent: full amount is "fees"
                }
                $balanceAmount = $amount < $effectiveFees ? $effectiveFees - $amount : 0;
                $advanceAmount = $amount > $effectiveFees ? $amount - $effectiveFees : 0;
                $baseSeq++;
                $receiptNo = 'R-' . now()->format('Ym') . '-' . str_pad((string) $baseSeq, 4, '0', STR_PAD_LEFT);

                Rent::query()->create([
                    'student_id' => $student->id,
                    'period_from' => $entry['period_from'],
                    'period_to' => $entry['period_to'],
                    'due_date' => $entry['due_date'],
                    'amount' => $amount,
                    'balance_amount' => $balanceAmount,
                    'advance_amount' => $advanceAmount,
                    'receipt_no' => $receiptNo,
                    'payment_mode' => $validated['payment_mode'],
                    'transaction_id' => $validated['transaction_id'] ?? null,
                    'cheque_number' => $validated['cheque_number'] ?? null,
                    'paid_at' => now(),
                ]);
                $lastBalance = $balanceAmount;
                $lastAdvance = $advanceAmount;
            }
        });

        return redirect()
            ->route('admin.rents.index')
            ->with('success', 'Rent recorded successfully for ' . $student->full_name . '.');
    }

    /**
     * Print receipt (amount in words).
     */
    public function receipt(Request $request, Rent $rent): View|JsonResponse
    {
        $rent->load(['student.bed', 'student.room', 'student.floor']);
        $amountInWords = $this->amountToWords((float) $rent->amount);
        
        // If AJAX request, return JSON with HTML content for modal
        if ($request->wantsJson() || $request->ajax()) {
            $html = view('admin.receipts._rent_content', [
                'rent' => $rent,
                'amountInWords' => $amountInWords,
            ])->render();
            $title = 'Rent Receipt - ' . ($rent->receipt_no ?? 'R' . $rent->id);

            return response()->json([
                'success' => true,
                'html' => $html,
                'title' => $title,
                'pdfDownloadUrl' => route('admin.rents.receipt.pdf', $rent),
                'receiptNo' => $rent->receipt_no ?? 'R' . $rent->id,
                'rentId' => $rent->id,
                'emailType' => 'rent',
                'emailId' => $rent->id,
            ]);
        }
        
        $pdfDownloadUrl = route('admin.rents.receipt.pdf', $rent);

        return view('admin.rents.receipt', [
            'rent' => $rent,
            'amountInWords' => $amountInWords,
            'pdfDownloadUrl' => $pdfDownloadUrl,
            'emailType' => 'rent',
            'emailId' => $rent->id,
        ]);
    }

    private function amountToWords(float $amount): string
    {
        $rupees = (int) floor($amount);
        $paise = (int) round(($amount - $rupees) * 100);
        $words = $this->numberToWords($rupees);
        $out = $words ? ucfirst($words) . ' Rupees' : 'Zero Rupees';
        if ($paise > 0) {
            $out .= ' and ' . ucfirst($this->numberToWords($paise)) . ' Paise';
        }
        return $out . ' only';
    }

    private function numberToWords(int $n): string
    {
        if ($n === 0) {
            return '';
        }
        $ones = ['', 'One', 'Two', 'Three', 'Four', 'Five', 'Six', 'Seven', 'Eight', 'Nine', 'Ten', 'Eleven', 'Twelve', 'Thirteen', 'Fourteen', 'Fifteen', 'Sixteen', 'Seventeen', 'Eighteen', 'Nineteen'];
        $tens = ['', '', 'Twenty', 'Thirty', 'Forty', 'Fifty', 'Sixty', 'Seventy', 'Eighty', 'Ninety'];
        if ($n < 20) {
            return $ones[$n];
        }
        if ($n < 100) {
            return trim($tens[(int) floor($n / 10)] . ' ' . $ones[$n % 10]);
        }
        if ($n < 1000) {
            $h = (int) floor($n / 100);
            $r = $n % 100;
            return trim($ones[$h] . ' Hundred ' . $this->numberToWords($r));
        }
        if ($n < 100000) {
            $th = (int) floor($n / 1000);
            $r = $n % 1000;
            return trim($this->numberToWords($th) . ' Thousand ' . $this->numberToWords($r));
        }
        if ($n < 10000000) {
            $l = (int) floor($n / 100000);
            $r = $n % 100000;
            return trim($this->numberToWords($l) . ' Lakh ' . $this->numberToWords($r));
        }
        $c = (int) floor($n / 10000000);
        $r = $n % 10000000;
        return trim($this->numberToWords($c) . ' Crore ' . $this->numberToWords($r));
    }
}
