<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Expense;
use App\Models\Rent;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ReportController extends Controller
{
    /**
     * Reports dashboard: daywise financial summary, charts, link to period report.
     */
    public function index(Request $request): View
    {
        $fromDate = $request->query('from_date') ?? now()->startOfMonth()->format('Y-m-d');
        $toDate = $request->query('to_date') ?? now()->format('Y-m-d');

        $rentTotal = Rent::query()
            ->whereDate('paid_at', '>=', $fromDate)
            ->whereDate('paid_at', '<=', $toDate)
            ->sum('amount');

        $depositTotal = Deposit::query()
            ->whereDate('actual_deposit_date', '>=', $fromDate)
            ->whereDate('actual_deposit_date', '<=', $toDate)
            ->sum('amount');

        $expenseTotal = Expense::query()
            ->whereDate('expense_date', '>=', $fromDate)
            ->whereDate('expense_date', '<=', $toDate)
            ->sum('amount');

        $revenueTotal = $rentTotal + $depositTotal;

        // Monthly trend (last 12 months): Rent, Deposit, Expense per month
        $startMonth = Carbon::parse($fromDate)->startOfMonth();
        $endMonth = Carbon::parse($toDate)->endOfMonth();
        $months = [];
        $rentByMonth = [];
        $depositByMonth = [];
        $expenseByMonth = [];
        $current = $startMonth->copy();
        while ($current->lte($endMonth)) {
            $key = $current->format('Y-m');
            $months[] = $current->format('M Y');
            $rentByMonth[] = (float) Rent::query()
                ->whereYear('paid_at', $current->year)
                ->whereMonth('paid_at', $current->month)
                ->sum('amount');
            $depositByMonth[] = (float) Deposit::query()
                ->whereYear('actual_deposit_date', $current->year)
                ->whereMonth('actual_deposit_date', $current->month)
                ->sum('amount');
            $expenseByMonth[] = (float) Expense::query()
                ->whereYear('expense_date', $current->year)
                ->whereMonth('expense_date', $current->month)
                ->sum('amount');
            $current->addMonth();
        }

        // Payment mode distribution (revenue: rent + deposit) in period
        $paymentModes = config('student_registration.payment_modes', []);
        $modeLabels = [];
        $modeAmounts = [];
        foreach (array_keys($paymentModes) as $mode) {
            $rentMode = (float) Rent::query()
                ->whereDate('paid_at', '>=', $fromDate)
                ->whereDate('paid_at', '<=', $toDate)
                ->where('payment_mode', $mode)
                ->sum('amount');
            $depositMode = (float) Deposit::query()
                ->whereDate('actual_deposit_date', '>=', $fromDate)
                ->whereDate('actual_deposit_date', '<=', $toDate)
                ->where('payment_mode', $mode)
                ->sum('amount');
            $total = $rentMode + $depositMode;
            if ($total > 0) {
                $modeLabels[] = $paymentModes[$mode] ?? $mode;
                $modeAmounts[] = round($total, 2);
            }
        }

        // Pie: revenue stream share (Rent vs Deposit)
        $pieLabels = ['Rent', 'Deposit'];
        $pieSeries = [round($rentTotal, 2), round($depositTotal, 2)];
        if ($revenueTotal <= 0) {
            $pieSeries = [0, 0];
        }

        return view('admin.reports.index', [
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'rentTotal' => $rentTotal,
            'depositTotal' => $depositTotal,
            'expenseTotal' => $expenseTotal,
            'revenueTotal' => $revenueTotal,
            'months' => $months,
            'rentByMonth' => $rentByMonth,
            'depositByMonth' => $depositByMonth,
            'expenseByMonth' => $expenseByMonth,
            'modeLabels' => $modeLabels,
            'modeAmounts' => $modeAmounts,
            'pieLabels' => $pieLabels,
            'pieSeries' => $pieSeries,
            'paymentModes' => $paymentModes,
        ]);
    }

    /**
     * Comprehensive period report: union of Rent, Deposit, Expense with filters.
     */
    public function period(Request $request): View
    {
        $fromDate = $request->query('from_date');
        $toDate = $request->query('to_date');
        $paymentType = $request->query('payment_type', 'all'); // rent, deposit, expense, all
        $paymentMode = $request->query('payment_mode');

        $rows = collect();
        $paymentModes = config('student_registration.payment_modes', []);

        if (in_array($paymentType, ['all', 'rent'])) {
            $rentQuery = Rent::query()
                ->with('student:id,first_name,last_name')
                ->select('id', 'student_id', 'paid_at', 'amount', 'payment_mode', 'receipt_no');
            if ($fromDate) {
                $rentQuery->whereDate('paid_at', '>=', $fromDate);
            }
            if ($toDate) {
                $rentQuery->whereDate('paid_at', '<=', $toDate);
            }
            if ($paymentMode !== null && $paymentMode !== '') {
                $rentQuery->where('payment_mode', $paymentMode);
            }
            foreach ($rentQuery->orderByDesc('paid_at')->orderByDesc('id')->get() as $r) {
                $rows->push((object)[
                    'date' => $r->paid_at ? Carbon::parse($r->paid_at)->format('Y-m-d') : null,
                    'date_display' => $r->paid_at ? Carbon::parse($r->paid_at)->format('d/m/Y') : '—',
                    'type' => 'Rent',
                    'description' => $r->student ? trim($r->student->first_name . ' ' . $r->student->last_name) : '—',
                    'amount' => $r->amount,
                    'payment_mode' => $r->payment_mode,
                    'reference' => $r->receipt_no ?? ('R' . $r->id),
                ]);
            }
        }

        if (in_array($paymentType, ['all', 'deposit'])) {
            $depositQuery = Deposit::query()
                ->with('student:id,first_name,last_name')
                ->select('id', 'student_id', 'actual_deposit_date', 'amount', 'payment_mode');
            if ($fromDate) {
                $depositQuery->whereDate('actual_deposit_date', '>=', $fromDate);
            }
            if ($toDate) {
                $depositQuery->whereDate('actual_deposit_date', '<=', $toDate);
            }
            if ($paymentMode !== null && $paymentMode !== '') {
                $depositQuery->where('payment_mode', $paymentMode);
            }
            foreach ($depositQuery->orderByDesc('actual_deposit_date')->orderByDesc('id')->get() as $d) {
                $rows->push((object)[
                    'date' => $d->actual_deposit_date?->format('Y-m-d'),
                    'date_display' => $d->actual_deposit_date?->format('d/m/Y') ?? '—',
                    'type' => 'Deposit',
                    'description' => $d->student ? trim($d->student->first_name . ' ' . $d->student->last_name) : '—',
                    'amount' => $d->amount,
                    'payment_mode' => $d->payment_mode,
                    'reference' => 'D' . $d->id,
                ]);
            }
        }

        if (in_array($paymentType, ['all', 'expense'])) {
            $expenseQuery = Expense::query()
                ->with('expenseHead:id,name')
                ->select('id', 'expense_head_id', 'expense_date', 'amount', 'payment_mode', 'item_name');
            if ($fromDate) {
                $expenseQuery->whereDate('expense_date', '>=', $fromDate);
            }
            if ($toDate) {
                $expenseQuery->whereDate('expense_date', '<=', $toDate);
            }
            if ($paymentMode !== null && $paymentMode !== '') {
                $expenseQuery->where('payment_mode', $paymentMode);
            }
            foreach ($expenseQuery->orderByDesc('expense_date')->orderByDesc('id')->get() as $e) {
                $rows->push((object)[
                    'date' => $e->expense_date?->format('Y-m-d'),
                    'date_display' => $e->expense_date?->format('d/m/Y') ?? '—',
                    'type' => 'Expense',
                    'description' => $e->item_name . ($e->expenseHead ? ' (' . $e->expenseHead->name . ')' : ''),
                    'amount' => - (float) $e->amount,
                    'payment_mode' => $e->payment_mode,
                    'reference' => $e->item_name,
                ]);
            }
        }

        $rows = $rows->sortByDesc(function ($r) {
            return $r->date ?? '0000-00-00';
        })->values();
        $total = $rows->sum('amount');

        return view('admin.reports.period', [
            'rows' => $rows,
            'total' => $total,
            'fromDate' => $fromDate,
            'toDate' => $toDate,
            'paymentType' => $paymentType,
            'paymentMode' => $paymentMode,
            'paymentModes' => $paymentModes,
        ]);
    }
}
