<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bed;
use App\Models\Floor;
use App\Models\Room;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class RoomController extends Controller
{
    public function index()
    {
        $rooms = Room::query()->with(['floor', 'beds'])->withCount('beds')->orderBy('floor_id')->orderBy('name')->get();
        return view('admin.master.rooms.index', compact('rooms'));
    }

    public function show(Room $room)
    {
        $room->load(['floor', 'beds']);
        return view('admin.master.rooms.show', compact('room'));
    }

    public function create()
    {
        $floors = Floor::query()->orderBy('sort_order')->orderBy('name')->get();
        return view('admin.master.rooms.create', compact('floors'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'floor_id' => ['required', 'exists:floors,id'],
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('rooms', 'name')->where('floor_id', $request->floor_id),
            ],
            'capacity' => ['nullable', 'integer', 'min:0'],
            'add_beds' => ['nullable', 'in:single,bulk'],
            'bed_name' => ['required_if:add_beds,single', 'nullable', 'string', 'max:255'],
            'bed_count' => ['required_if:add_beds,bulk', 'nullable', 'integer', 'min:1', 'max:50'],
            'bed_prefix' => ['nullable', 'string', 'max:100'],
        ]);

        $capacity = (int) ($validated['capacity'] ?? 0);
        $room = Room::query()->create([
            'floor_id' => $validated['floor_id'],
            'name' => $validated['name'],
            'capacity' => $capacity,
        ]);

        $floor = $room->floor;
        $floorBedsCount = $floor->rooms()->withCount('beds')->get()->sum('beds_count');
        $newBedsCount = 0;

        if ($request->input('add_beds') === 'single' && !empty(trim($request->input('bed_name')))) {
            $newBedsCount = 1;
            if ($capacity > 0 && $newBedsCount > $capacity) {
                $room->delete();
                return back()->withInput()->withErrors(['add_beds' => "Room capacity is {$capacity}. Cannot add more than {$capacity} bed(s)."]);
            }
            if ($floor->capacity > 0 && ($floorBedsCount + $newBedsCount) > $floor->capacity) {
                $room->delete();
                return back()->withInput()->withErrors(['add_beds' => "Floor capacity is {$floor->capacity}. Cannot add more beds to this floor."]);
            }
            Bed::query()->create([
                'room_id' => $room->id,
                'name' => trim($request->bed_name),
                'status' => Bed::STATUS_AVAILABLE,
            ]);
        } elseif ($request->input('add_beds') === 'bulk' && $request->input('bed_count')) {
            $count = (int) $request->bed_count;
            $newBedsCount = $count;
            if ($capacity > 0 && $newBedsCount > $capacity) {
                $room->delete();
                return back()->withInput()->withErrors(['bed_count' => "Room capacity is {$capacity}. Cannot add more than {$capacity} bed(s)."]);
            }
            if ($floor->capacity > 0 && ($floorBedsCount + $newBedsCount) > $floor->capacity) {
                $room->delete();
                return back()->withInput()->withErrors(['add_beds' => "Floor capacity is {$floor->capacity}. Cannot add more beds to this floor."]);
            }
            $prefix = trim($request->input('bed_prefix', 'Bed'));
            for ($i = 1; $i <= $count; $i++) {
                Bed::query()->create([
                    'room_id' => $room->id,
                    'name' => $prefix . ' ' . $i,
                    'status' => Bed::STATUS_AVAILABLE,
                ]);
            }
        }

        $message = 'Room created successfully.';
        if ($newBedsCount > 0) {
            $message .= " {$newBedsCount} bed(s) added.";
        }

        return redirect()
            ->route('admin.rooms.index')
            ->with('success', $message);
    }

    public function edit(Room $room)
    {
        $floors = Floor::query()->orderBy('sort_order')->orderBy('name')->get();
        return view('admin.master.rooms.edit', compact('room', 'floors'));
    }

    public function update(Request $request, Room $room)
    {
        $validated = $request->validate([
            'floor_id' => ['required', 'exists:floors,id'],
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('rooms', 'name')->where('floor_id', $request->floor_id)->ignore($room->id),
            ],
            'capacity' => ['nullable', 'integer', 'min:0'],
        ]);
        $room->update([
            'floor_id' => $validated['floor_id'],
            'name' => $validated['name'],
            'capacity' => (int) ($validated['capacity'] ?? 0),
        ]);
        return redirect()->route('admin.rooms.index')->with('success', 'Room updated successfully.');
    }

    public function destroy(Room $room)
    {
        $room->delete();
        return redirect()
            ->route('admin.rooms.index')
            ->with('success', 'Room deleted successfully.');
    }
}
