<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AgreementHistory;
use App\Models\Bed;
use App\Models\Deposit;
use App\Models\Rent;
use App\Models\Student;
use App\Models\StudentBedHistory;
use App\Models\StudentDocument;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class StudentRegistrationController extends Controller
{
    public function index(Request $request)
    {
        $filters = config('student_registration.list_filters', []);
        $activeFilter = $request->query('filter', 'active');
        if (! array_key_exists($activeFilter, $filters)) {
            $activeFilter = 'active';
        }

        $counts = array_merge(
            array_fill_keys(array_keys($filters), 0),
            [
                'active' => Student::query()->where('status', Student::STATUS_ACTIVE)->count(),
                'pending' => Student::query()->where('status', Student::STATUS_PENDING)->count(),
                'alumna' => Student::query()->where('status', Student::STATUS_LEFT)->count(),
                'not_eligible' => Student::query()->where('status', Student::STATUS_CANCELLED)->count(),
                'information_pending' => Student::query()->whereIn('form_status', [
                    Student::FORM_STATUS_PENDING,
                    Student::FORM_STATUS_DOCUMENTS_UPLOADED,
                    Student::FORM_STATUS_BED_ALLOTTED,
                    Student::FORM_STATUS_DEPOSIT_PAID,
                ])->count(),
            ]
        );

        $totalStudents = Student::query()->count();

        return view('admin.students.registration.index', compact('filters', 'activeFilter', 'counts', 'totalStudents'));
    }

    /**
     * Export students list (CSV / Excel). Uses same filter as list. PDF triggers print.
     */
    public function export(Request $request)
    {
        $filter = $request->query('filter', 'active');
        $filters = config('student_registration.list_filters', []);
        if (! array_key_exists($filter, $filters)) {
            $filter = 'active';
        }
        $format = $request->query('format', 'csv');
        if (! in_array($format, ['csv', 'excel'], true)) {
            $format = 'csv';
        }

        $query = Student::query();
        match ($filter) {
            'active' => $query->where('status', Student::STATUS_ACTIVE),
            'pending' => $query->where('status', Student::STATUS_PENDING),
            'alumna' => $query->where('status', Student::STATUS_LEFT),
            'not_eligible' => $query->where('status', Student::STATUS_CANCELLED),
            'information_pending' => $query->whereIn('form_status', [
                Student::FORM_STATUS_PENDING,
                Student::FORM_STATUS_DOCUMENTS_UPLOADED,
                Student::FORM_STATUS_BED_ALLOTTED,
                Student::FORM_STATUS_DEPOSIT_PAID,
            ]),
            default => $query->where('status', Student::STATUS_ACTIVE),
        };
        $students = $query->orderBy('first_name')->orderBy('last_name')->get();

        $headers = ['#', 'Full Name', 'Email', 'Mobile', 'Status', 'Form Progress', 'Room / Bed', 'Updated'];
        $rows = [];
        foreach ($students as $s) {
            $rows[] = [
                $s->id,
                $s->full_name,
                $s->email ?? '',
                $s->student_mobile ?? '',
                config('student_registration.status.' . $s->status, $s->status),
                config('student_registration.form_status.' . $s->form_status, $s->form_status),
                $s->room_bed_identifier ?? '',
                $s->updated_at?->format('Y-m-d') ?? '',
            ];
        }

        $filename = 'students-' . $filter . '-' . now()->format('Y-m-d');
        if ($format === 'excel') {
            $filename .= '.xls';
            $contentType = 'application/vnd.ms-excel';
        } else {
            $filename .= '.csv';
            $contentType = 'text/csv';
        }

        $callback = function () use ($headers, $rows, $format) {
            $out = fopen('php://output', 'w');
            if ($format === 'csv' || $format === 'excel') {
                fputcsv($out, $headers);
                foreach ($rows as $row) {
                    fputcsv($out, $row);
                }
            }
            fclose($out);
        };

        return response()->stream($callback, 200, [
            'Content-Type' => $contentType,
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    public function show(Student $student): View
    {
        $student->load([
            'floor', 'room', 'bed',
            'documents',
            'agreementHistories',
            'remarks',
            'deposits',
            'rents',
            'bedHistory.floor', 'bedHistory.room', 'bedHistory.bed',
        ]);
        return view('admin.students.show', compact('student'));
    }

    public function edit(Student $student): View
    {
        $streams = config('student_registration.streams', []);
        return view('admin.students.edit', compact('student', 'streams'));
    }

    public function update(Request $request, Student $student): RedirectResponse
    {
        $validated = $request->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'middle_name' => ['nullable', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'student_mobile' => ['nullable', 'string', 'max:20'],
            'parent_mobile' => ['nullable', 'string', 'max:20'],
            'permanent_locality' => ['nullable', 'string', 'max:255'],
            'permanent_district' => ['nullable', 'string', 'max:255'],
            'permanent_state' => ['nullable', 'string', 'max:255'],
        ]);
        $student->update([
            'first_name' => strtoupper($validated['first_name']),
            'middle_name' => $validated['middle_name'] ? strtoupper($validated['middle_name']) : null,
            'last_name' => strtoupper($validated['last_name']),
            'email' => $validated['email'] ?? null,
            'student_mobile' => $validated['student_mobile'] ?? null,
            'parent_mobile' => $validated['parent_mobile'] ?? null,
            'permanent_locality' => $validated['permanent_locality'] ?? null,
            'permanent_district' => $validated['permanent_district'] ?? null,
            'permanent_state' => $validated['permanent_state'] ?? null,
        ]);
        return redirect()->route('admin.students.show', $student)->with('success', 'Student updated.');
    }

    public function release(Request $request, Student $student): RedirectResponse
    {
        $student->update(['status' => Student::STATUS_LEFT]);
        $student->bed?->update(['student_id' => null, 'status' => Bed::STATUS_AVAILABLE]);
        return redirect()->route('admin.students.index')->with('success', 'Student released.');
    }

    /**
     * AJAX endpoint for DataTables server-side. Returns JSON: draw, recordsTotal, recordsFiltered, data.
     */
    public function data(Request $request): JsonResponse
    {
        $draw = (int) $request->query('draw', 1);
        $start = (int) $request->query('start', 0);
        $length = (int) $request->query('length', 10);
        $length = min(max($length, 1), 100);
        $searchValue = $request->query('search')['value'] ?? '';
        $searchValue = is_string($searchValue) ? trim($searchValue) : '';
        $orderColumnIndex = (int) ($request->query('order')[0]['column'] ?? 0);
        $orderDir = ($request->query('order')[0]['dir'] ?? 'desc') === 'asc' ? 'asc' : 'desc';
        $filter = $request->query('filter', 'active');
        $filters = config('student_registration.list_filters', []);
        if (! array_key_exists($filter, $filters)) {
            $filter = 'active';
        }

        $query = Student::query();
        match ($filter) {
            'active' => $query->where('status', Student::STATUS_ACTIVE),
            'pending' => $query->where('status', Student::STATUS_PENDING),
            'alumna' => $query->where('status', Student::STATUS_LEFT),
            'not_eligible' => $query->where('status', Student::STATUS_CANCELLED),
            'information_pending' => $query->whereIn('form_status', [
                Student::FORM_STATUS_PENDING,
                Student::FORM_STATUS_DOCUMENTS_UPLOADED,
                Student::FORM_STATUS_BED_ALLOTTED,
                Student::FORM_STATUS_DEPOSIT_PAID,
            ]),
            default => $query->where('status', Student::STATUS_ACTIVE),
        };

        $recordsTotal = (clone $query)->count();

        if ($searchValue !== '') {
            $term = '%' . str_replace(['%', '_'], ['\\%', '\\_'], $searchValue) . '%';
            $query->where(function ($q) use ($term) {
                $q->where('students.first_name', 'like', $term)
                    ->orWhere('students.middle_name', 'like', $term)
                    ->orWhere('students.last_name', 'like', $term)
                    ->orWhere('students.email', 'like', $term)
                    ->orWhere('students.student_mobile', 'like', $term)
                    ->orWhere('students.parent_mobile', 'like', $term)
                    ->orWhere('students.room_bed_identifier', 'like', $term)
                    ->orWhere('students.aadhar_number', 'like', $term);
            });
        }

        $recordsFiltered = $query->count();

        $orderColumns = [
            2 => 'students.first_name',
            3 => 'students.student_mobile',
            4 => 'students.room_bed_identifier',
            5 => 'students.created_at',
        ];
        if (isset($orderColumns[$orderColumnIndex])) {
            if ($orderColumnIndex === 2) {
                $query->orderBy('students.first_name', $orderDir)->orderBy('students.last_name', $orderDir);
            } else {
                $query->orderBy($orderColumns[$orderColumnIndex], $orderDir);
            }
        } else {
            $query->orderBy('students.updated_at', 'desc');
        }

        $students = $query->with(['documents', 'agreementHistories', 'rents'])
            ->skip($start)->take($length)->get();

        $data = [];
        foreach ($students as $index => $student) {
            $firstAgreement = $student->agreementHistories->sortBy('from_date')->first();
            $latestRent = $student->rents->sortByDesc('paid_at')->first();
            $profileDoc = $student->documents->where('document_type', 'profile_photo')->first();
            $profileUrl = $profileDoc ? Storage::url($profileDoc->file_path) : null;

            $profileImg = $profileDoc
                ? '<img src="' . e(asset('storage/' . $profileDoc->file_path)) . '" alt="" class="rounded-circle" style="width:36px;height:36px;object-fit:cover;">'
                : '<span class="avatar avatar-sm rounded-circle bg-label-secondary">' . strtoupper(mb_substr($student->first_name ?? '', 0, 1)) . '</span>';
            $data[] = [
                'sr_no' => $start + $index + 1,
                'profile' => $profileImg,
                'full_name' => $student->full_name,
                'contact' => $student->student_mobile ?? $student->email ?? '—',
                'room_bed_identifier' => $student->room_bed_identifier ?? '—',
                'admission_date' => $firstAgreement?->from_date?->format('d/m/Y') ?? $student->created_at?->format('d/m/Y') ?? '—',
                'leaving_date' => $firstAgreement?->to_date?->format('d/m/Y') ?? '—',
                'rent' => $latestRent ? (string) $latestRent->amount : '—',
                'remark' => '—',
                'actions' => $this->buildActionDropdownHtml($student),
            ];
        }

        return response()->json([
            'draw' => $draw,
            'recordsTotal' => $recordsTotal,
            'recordsFiltered' => $recordsFiltered,
            'data' => $data,
        ]);
    }

    protected function buildActionDropdownHtml(Student $student): string
    {
        $viewUrl = route('admin.students.show', $student);
        $editUrl = route('admin.students.edit', $student);
        $docUrl = route('admin.students.registration.step2', $student);
        $addRentUrl = route('admin.rents.create') . '?student_id=' . $student->id;
        $agreementUrl = route('admin.agreements.index') . '?student_id=' . $student->id;
        $releaseUrl = route('admin.students.release', $student);
        $paymentUrl = route('admin.students.show', $student) . '#payment-history';
        $roomHistoryUrl = route('admin.students.show', $student) . '#room-history';
        $addRemarkUrl = route('admin.students.show', $student) . '#remarks';
        $roomTransferUrl = route('admin.students.show', $student) . '#room-history';

        $items = [
            '<li><a class="dropdown-item" href="' . e($viewUrl) . '"><i class="ri-eye-line me-2"></i>View Details</a></li>',
            '<li><a class="dropdown-item" href="' . e($addRentUrl) . '"><i class="ri-money-dollar-circle-line me-2"></i>Add Rent</a></li>',
            '<li><a class="dropdown-item" href="' . e($addRemarkUrl) . '"><i class="ri-chat-3-line me-2"></i>Add Remark</a></li>',
            '<li><a class="dropdown-item" href="' . e($roomTransferUrl) . '"><i class="ri-home-4-line me-2"></i>Room Transfer</a></li>',
            '<li><a class="dropdown-item" href="' . e($docUrl) . '"><i class="ri-file-list-line me-2"></i>Student Document</a></li>',
            '<li><a class="dropdown-item" href="' . e($paymentUrl) . '"><i class="ri-history-line me-2"></i>Payment History</a></li>',
            '<li><a class="dropdown-item" href="' . e($roomHistoryUrl) . '"><i class="ri-home-4-line me-2"></i>Room History</a></li>',
            '<li><a class="dropdown-item" href="' . e($editUrl) . '"><i class="ri-edit-line me-2"></i>Edit Student Details</a></li>',
        ];
        if ($student->status === Student::STATUS_ACTIVE) {
            $items[] = '<li><form method="POST" action="' . e($releaseUrl) . '" class="d-inline" onsubmit="return confirm(\'Release this student?\');"><input type="hidden" name="_token" value="' . e(csrf_token()) . '"><button type="submit" class="dropdown-item text-danger border-0 bg-transparent w-100 text-start"><i class="ri-user-unfollow-line me-2"></i>Release Student</button></form></li>';
        }
        $items[] = '<li><a class="dropdown-item" href="' . e($agreementUrl) . '"><i class="ri-file-text-line me-2"></i>Agreement History</a></li>';

        return '<div class="dropdown"><button class="btn btn-sm btn-primary dropdown-toggle" type="button" data-bs-toggle="dropdown" aria-expanded="false">Action</button><ul class="dropdown-menu dropdown-menu-end">' . implode('', $items) . '</ul></div>';
    }

    public function create()
    {
        $streams = config('student_registration.streams', []);
        $students = Student::query()->where('status', Student::STATUS_ACTIVE)->orderBy('first_name')->get(['id', 'first_name', 'middle_name', 'last_name']);
        return view('admin.students.registration.step1', compact('streams', 'students'));
    }

    public function storeStep1(Request $request)
    {
        $validated = $request->validate([
            'first_name' => ['required', 'string', 'max:255'],
            'middle_name' => ['nullable', 'string', 'max:255'],
            'last_name' => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255'],
            'date_of_birth' => ['nullable', 'date'],
            'permanent_locality' => ['nullable', 'string', 'max:255'],
            'permanent_taluka' => ['nullable', 'string', 'max:255'],
            'permanent_district' => ['nullable', 'string', 'max:255'],
            'permanent_state' => ['nullable', 'string', 'max:255'],
            'student_mobile' => ['nullable', 'string', 'max:20'],
            'parent_mobile' => ['nullable', 'string', 'max:20'],
            'aadhar_number' => ['nullable', 'string', 'max:20'],
            'standard_year' => ['nullable', 'string', 'max:100'],
            'stream' => ['nullable', 'string', 'max:50'],
            'college_name' => ['nullable', 'string', 'max:255'],
            'college_in_time' => ['nullable', 'date_format:H:i'],
            'college_out_time' => ['nullable', 'date_format:H:i'],
            'reference_name' => ['nullable', 'string', 'max:255'],
            'reference_relation' => ['nullable', 'string', 'max:100'],
            'reference_contact' => ['nullable', 'string', 'max:20'],
            'reference_by_student_id' => ['nullable', 'exists:students,id'],
            'terms_accepted' => ['required', 'accepted'],
        ]);

        $duplicate = Student::query()
            ->where('status', Student::STATUS_ACTIVE)
            ->where('first_name', $request->first_name)
            ->where('last_name', $request->last_name)
            ->when($request->middle_name, fn ($q) => $q->where('middle_name', $request->middle_name))
            ->when(!$request->middle_name, fn ($q) => $q->whereNull('middle_name'))
            ->first();
        if ($duplicate) {
            return back()->withInput()->withErrors(['first_name' => 'An active student with this name already exists.']);
        }

        $student = Student::query()->create([
            'first_name' => strtoupper($validated['first_name']),
            'middle_name' => $validated['middle_name'] ? strtoupper($validated['middle_name']) : null,
            'last_name' => strtoupper($validated['last_name']),
            'email' => $validated['email'] ?? null,
            'date_of_birth' => $validated['date_of_birth'] ?? null,
            'permanent_locality' => $validated['permanent_locality'] ?? null,
            'permanent_taluka' => $validated['permanent_taluka'] ?? null,
            'permanent_district' => $validated['permanent_district'] ?? null,
            'permanent_state' => $validated['permanent_state'] ?? null,
            'student_mobile' => $validated['student_mobile'] ?? null,
            'parent_mobile' => $validated['parent_mobile'] ?? null,
            'aadhar_number' => $validated['aadhar_number'] ?? null,
            'standard_year' => $validated['standard_year'] ?? null,
            'stream' => $validated['stream'] ?? null,
            'college_name' => $validated['college_name'] ?? null,
            'college_in_time' => $validated['college_in_time'] ?? null,
            'college_out_time' => $validated['college_out_time'] ?? null,
            'reference_name' => $validated['reference_name'] ?? null,
            'reference_relation' => $validated['reference_relation'] ?? null,
            'reference_contact' => $validated['reference_contact'] ?? null,
            'reference_by_student_id' => $validated['reference_by_student_id'] ?? null,
            'terms_accepted_at' => now(),
            'form_status' => Student::FORM_STATUS_PENDING,
            'status' => Student::STATUS_PENDING,
        ]);

        return redirect()->route('admin.students.registration.step2', $student);
    }

    public function step2(Student $student)
    {
        // Check if student has already completed registration
        if ($student->status === Student::STATUS_ACTIVE || $student->form_status === Student::FORM_STATUS_ACTIVE) {
            return redirect()
                ->route('admin.students.show', $student)
                ->with('info', 'This student has already completed registration. You are viewing their profile.');
        }
        
        $this->ensureStep($student, Student::FORM_STATUS_PENDING);
        $documentTypes = config('student_registration.document_types', []);
        $uploadConfig = config('student_registration.document_upload', []);
        return view('admin.students.registration.step2', compact('student', 'documentTypes', 'uploadConfig'));
    }

    public function storeStep2(Request $request, Student $student)
    {
        $this->ensureStep($student, Student::FORM_STATUS_PENDING);
        $documentTypes = array_keys(config('student_registration.document_types', []));
        $rules = [];
        foreach ($documentTypes as $type) {
            $rules[$type] = ['required', 'file', 'mimes:jpg,jpeg,png,pdf', 'max:5120'];
        }
        $validated = $request->validate($rules);

        $disk = 'public';
        $basePath = 'student-documents/' . $student->id;
        foreach ($validated as $documentType => $file) {
            $originalName = $file->getClientOriginalName();
            $filename = uniqid() . '_' . time() . '_' . preg_replace('/[^a-zA-Z0-9._-]/', '_', $originalName);
            $path = $file->storeAs($basePath, $filename, $disk);
            StudentDocument::query()->updateOrCreate(
                ['student_id' => $student->id, 'document_type' => $documentType],
                [
                    'file_path' => $path,
                    'original_name' => $originalName,
                    'mime_type' => $file->getMimeType(),
                    'file_size' => $file->getSize(),
                ]
            );
        }

        $student->update(['form_status' => Student::FORM_STATUS_DOCUMENTS_UPLOADED]);
        return redirect()->route('admin.students.registration.step3', $student);
    }

    public function step3(Student $student)
    {
        // Check if student has already completed registration
        if ($student->status === Student::STATUS_ACTIVE || $student->form_status === Student::FORM_STATUS_ACTIVE) {
            return redirect()
                ->route('admin.students.show', $student)
                ->with('info', 'This student has already completed registration. You are viewing their profile.');
        }
        
        $this->ensureStep($student, Student::FORM_STATUS_DOCUMENTS_UPLOADED);
        $floors = \App\Models\Floor::query()->orderBy('sort_order')->orderBy('name')->get(['id', 'name']);
        return view('admin.students.registration.step3', compact('student', 'floors'));
    }

    public function storeStep3(Request $request, Student $student)
    {
        $this->ensureStep($student, Student::FORM_STATUS_DOCUMENTS_UPLOADED);
        $validated = $request->validate([
            'floor_id' => ['required', 'exists:floors,id'],
            'room_id' => ['required', 'exists:rooms,id'],
            'bed_id' => ['required', 'exists:beds,id'],
        ]);

        $bed = Bed::query()
            ->where('id', $validated['bed_id'])
            ->where('room_id', $validated['room_id'])
            ->where('status', Bed::STATUS_AVAILABLE)
            ->whereNull('student_id')
            ->firstOrFail();

        DB::transaction(function () use ($student, $bed, $validated) {
            $bed->update([
                'student_id' => $student->id,
                'status' => Bed::STATUS_OCCUPIED,
            ]);
            $room = $bed->room;
            $floor = $room->floor;
            $roomBedIdentifier = $room->name . $bed->name;
            $student->update([
                'floor_id' => $floor->id,
                'room_id' => $room->id,
                'bed_id' => $bed->id,
                'room_bed_identifier' => $roomBedIdentifier,
                'form_status' => Student::FORM_STATUS_BED_ALLOTTED,
            ]);
            StudentBedHistory::query()->create([
                'student_id' => $student->id,
                'floor_id' => $floor->id,
                'room_id' => $room->id,
                'bed_id' => $bed->id,
                'allotted_at' => now(),
            ]);
        });

        return redirect()->route('admin.students.registration.step4', $student);
    }

    public function step4(Student $student)
    {
        // Check if student has already completed registration
        if ($student->status === Student::STATUS_ACTIVE || $student->form_status === Student::FORM_STATUS_ACTIVE) {
            return redirect()
                ->route('admin.students.show', $student)
                ->with('info', 'This student has already completed registration. You are viewing their profile.');
        }
        
        $this->ensureStep($student, Student::FORM_STATUS_BED_ALLOTTED);
        $paymentModes = config('student_registration.payment_modes', []);
        return view('admin.students.registration.step4', compact('student', 'paymentModes'));
    }

    public function storeStep4(Request $request, Student $student)
    {
        $this->ensureStep($student, Student::FORM_STATUS_BED_ALLOTTED);
        $paymentModes = array_keys(config('student_registration.payment_modes', []));
        $validated = $request->validate([
            'from_date' => ['required', 'date'],
            'to_date' => ['required', 'date', 'after_or_equal:from_date'],
            'actual_deposit_date' => ['required', 'date'],
            'amount' => ['required', 'numeric', 'min:0'],
            'payment_mode' => ['required', Rule::in($paymentModes)],
            'transaction_id' => ['nullable', 'string', 'max:255'],
        ]);

        DB::transaction(function () use ($student, $validated) {
            Deposit::query()->create([
                'student_id' => $student->id,
                'from_date' => $validated['from_date'],
                'to_date' => $validated['to_date'],
                'actual_deposit_date' => $validated['actual_deposit_date'],
                'amount' => $validated['amount'],
                'payment_mode' => $validated['payment_mode'],
                'transaction_id' => $validated['transaction_id'] ?? null,
            ]);
            AgreementHistory::query()->create([
                'student_id' => $student->id,
                'from_date' => $validated['from_date'],
                'to_date' => $validated['to_date'],
            ]);
            $student->update(['form_status' => Student::FORM_STATUS_DEPOSIT_PAID]);
        });

        return redirect()->route('admin.students.registration.step5', $student);
    }

    public function step5(Student $student)
    {
        // Check if student has already completed registration
        if ($student->status === Student::STATUS_ACTIVE || $student->form_status === Student::FORM_STATUS_ACTIVE) {
            return redirect()
                ->route('admin.students.show', $student)
                ->with('info', 'This student has already completed registration. You are viewing their profile.');
        }
        
        $this->ensureStep($student, Student::FORM_STATUS_DEPOSIT_PAID);
        $paymentModes = config('student_registration.payment_modes', []);
        return view('admin.students.registration.step5', compact('student', 'paymentModes'));
    }

    public function storeStep5(Request $request, Student $student)
    {
        $this->ensureStep($student, Student::FORM_STATUS_DEPOSIT_PAID);
        $paymentModes = array_keys(config('student_registration.payment_modes', []));
        $validated = $request->validate([
            'period_from' => ['required', 'date'],
            'period_to' => ['required', 'date', 'after_or_equal:period_from'],
            'due_date' => ['required', 'date'],
            'amount' => ['required', 'numeric', 'min:0'],
            'payment_mode' => ['required', Rule::in($paymentModes)],
            'transaction_id' => ['nullable', 'string', 'max:255'],
        ]);

        DB::transaction(function () use ($student, $validated) {
            Rent::query()->create([
                'student_id' => $student->id,
                'period_from' => $validated['period_from'],
                'period_to' => $validated['period_to'],
                'due_date' => $validated['due_date'],
                'amount' => $validated['amount'],
                'payment_mode' => $validated['payment_mode'],
                'transaction_id' => $validated['transaction_id'] ?? null,
                'paid_at' => now(),
            ]);
            $student->update([
                'form_status' => Student::FORM_STATUS_ACTIVE,
                'status' => Student::STATUS_ACTIVE,
                'deposit_status' => true,
            ]);

            $points = (int) config('student_registration.referral_reward_points', 50);
            if ($points > 0 && $student->reference_by_student_id) {
                Student::query()->where('id', $student->reference_by_student_id)->increment('reward_points', $points);
            }
        });

        if (config('student_registration.sms_enabled', false) && $student->student_mobile) {
            $this->sendWelcomeSms($student);
        }

        return redirect()
            ->route('admin.students.index')
            ->with('success', 'Student registered successfully. Welcome SMS sent (if configured).');
    }

    protected function ensureStep(Student $student, string $expectedStatus): void
    {
        if ($student->form_status !== $expectedStatus) {
            abort(403, 'Invalid step. Student is currently at: ' . ($student->form_status ?? 'unknown') . ', expected: ' . $expectedStatus);
        }
    }

    protected function sendWelcomeSms(Student $student): void
    {
        $message = config('student_registration.sms_welcome_message', 'Welcome to hostel. Your room: %s, bed: %s.');
        $message = sprintf($message, $student->room->name ?? '', $student->bed->name ?? '');
        // Placeholder: integrate with SMS gateway via config (e.g. HTTP API URL, credentials in .env)
        logger()->info('SMS would be sent', ['to' => $student->student_mobile, 'message' => $message]);
    }
}
