<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Bed;
use App\Models\Student;
use App\Models\StudentBedHistory;
use Carbon\Carbon;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class StudentSwitchController extends Controller
{
    /**
     * Switch / reallocate form: effectivity date + bulk rows (student → target bed).
     */
    public function index(): View
    {
        $students = Student::query()
            ->with(['bed.room.floor'])
            ->where('status', Student::STATUS_ACTIVE)
            ->where('form_status', Student::FORM_STATUS_ACTIVE)
            ->whereNotNull('bed_id')
            ->orderBy('first_name')
            ->orderBy('last_name')
            ->get(['id', 'first_name', 'middle_name', 'last_name', 'bed_id', 'room_id', 'floor_id', 'room_bed_identifier']);

        $availableBeds = Bed::query()
            ->with(['room.floor'])
            ->whereNull('student_id')
            ->orderBy('id')
            ->get()
            ->map(function (Bed $bed) {
                $room = $bed->room;
                $floor = $room?->floor;
                $label = ($floor ? $floor->name . ' - ' : '') . ($room ? $room->name . ' - ' : '') . $bed->name;
                return ['id' => $bed->id, 'label' => $label, 'room_id' => $bed->room_id, 'floor_id' => $room?->floor_id];
            });

        return view('admin.students.switch.index', [
            'students' => $students,
            'availableBeds' => $availableBeds,
        ]);
    }

    /**
     * Store bulk switch: for each row, move student to target bed and log history.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'effectivity_date' => ['required', 'date'],
            'switches' => ['required', 'array', 'min:1'],
            'switches.*.student_id' => ['nullable', 'integer', 'exists:students,id'],
            'switches.*.target_bed_id' => ['nullable', 'integer', 'exists:beds,id'],
        ], [], [
            'switches.*.student_id' => 'student',
            'switches.*.target_bed_id' => 'target bed',
        ]);

        $effectivity = Carbon::parse($validated['effectivity_date'])->startOfDay();
        $processed = 0;
        $errors = [];

        foreach ($validated['switches'] as $row) {
            $studentId = (int) $row['student_id'];
            $targetBedId = (int) $row['target_bed_id'];
            if (! $studentId || ! $targetBedId) {
                continue;
            }

            $student = Student::query()->with('bed.room.floor')->find($studentId);
            $targetBed = Bed::query()->with('room.floor')->find($targetBedId);

            if (! $student || ! $targetBed) {
                continue;
            }

            if (! $student->bed_id) {
                $errors[] = "Student {$student->full_name} has no current bed.";
                continue;
            }

            if ($student->bed_id === $targetBedId) {
                $errors[] = "Student {$student->full_name} is already on the selected bed.";
                continue;
            }

            if ($targetBed->student_id && $targetBed->student_id !== $studentId) {
                $errors[] = "Target bed {$targetBed->room->name}/{$targetBed->name} is occupied. Choose a vacant bed.";
                continue;
            }

            DB::transaction(function () use ($student, $targetBed, $effectivity) {
                $oldBed = $student->bed;
                $room = $targetBed->room;
                $floor = $room->floor;

                if ($oldBed) {
                    $oldBed->update(['student_id' => null, 'status' => Bed::STATUS_AVAILABLE]);
                    StudentBedHistory::query()
                        ->where('student_id', $student->id)
                        ->whereNull('left_at')
                        ->update(['left_at' => $effectivity]);
                }

                $targetBed->update(['student_id' => $student->id, 'status' => Bed::STATUS_OCCUPIED]);

                $roomBedIdentifier = $room->name . $targetBed->name;
                $student->update([
                    'floor_id' => $floor->id,
                    'room_id' => $room->id,
                    'bed_id' => $targetBed->id,
                    'room_bed_identifier' => $roomBedIdentifier,
                ]);

                StudentBedHistory::query()->create([
                    'student_id' => $student->id,
                    'floor_id' => $floor->id,
                    'room_id' => $room->id,
                    'bed_id' => $targetBed->id,
                    'allotted_at' => $effectivity,
                ]);
            });

            $processed++;
        }

        if (! empty($errors)) {
            return redirect()
                ->route('admin.students.switch')
                ->withInput()
                ->with('error', implode(' ', $errors));
        }

        if ($processed === 0) {
            return redirect()
                ->route('admin.students.switch')
                ->with('error', 'No valid switch entries. Check student and target bed.');
        }

        return redirect()
            ->route('admin.students.switch')
            ->with('success', $processed . ' student(s) switched successfully. Effectivity: ' . $effectivity->format('d/m/Y'));
    }
}
