<?php

namespace App\Http\Controllers\User\Auth;

use App\Http\Controllers\Controller;
use App\Services\OtpService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Validation\ValidationException;

class VerifyOtpController extends Controller
{
    public function __construct(
        protected OtpService $otpService
    ) {}

    public function showVerifyForm(Request $request)
    {
        if (!Auth::guard('user')->check()) {
            return redirect()->route('user.login');
        }
        $user = Auth::guard('user')->user();
        $maskedEmail = OtpService::maskEmail($user->email);
        $devOtp = app()->environment('local') ? config('otp.static_otp', '123456') : null;
        return view('auth.user.verify-otp', compact('maskedEmail', 'devOtp'));
    }

    public function verify(Request $request)
    {
        $request->validate([
            'otp' => ['required', 'string', 'size:6'],
        ]);

        $key = 'otp-verify-user-' . (Auth::guard('user')->id() ?? $request->ip());
        if (RateLimiter::tooManyAttempts($key, 5)) {
            $seconds = RateLimiter::availableIn($key);
            throw ValidationException::withMessages([
                'otp' => ['Too many attempts. Please try again in ' . $seconds . ' seconds.'],
            ]);
        }

        $user = Auth::guard('user')->user();
        if (!$user) {
            return redirect()->route('user.login');
        }

        if (!$this->otpService->verify($user, 'user', $request->otp)) {
            RateLimiter::hit($key);
            throw ValidationException::withMessages([
                'otp' => [__('The verification code is invalid or has expired.')],
            ]);
        }

        RateLimiter::clear($key);
        session(['otp_verified_user' => true]);

        return redirect()->intended(route('user.dashboard'));
    }

    public function resend(Request $request)
    {
        $key = 'otp-resend-user-' . (Auth::guard('user')->id() ?? $request->ip());
        if (RateLimiter::tooManyAttempts($key, 3)) {
            $seconds = RateLimiter::availableIn($key);
            return back()->with('error', __('Please wait :seconds seconds before requesting a new code.', ['seconds' => $seconds]));
        }

        $user = Auth::guard('user')->user();
        if (!$user) {
            return redirect()->route('user.login');
        }

        $this->otpService->generate($user, 'user');
        RateLimiter::hit($key);

        return back()->with('status', __('A new verification code has been sent to your email.'));
    }
}
