<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class AdminNotification extends Model
{
    protected $fillable = [
        'type',
        'priority',
        'related_id',
        'related_type',
        'title',
        'message',
        'meta',
        'action_url',
        'is_read',
        'read_at',
    ];

    protected $casts = [
        'meta' => 'array',
        'is_read' => 'boolean',
        'read_at' => 'datetime',
    ];

    // Scopes
    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }

    public function scopeRead($query)
    {
        return $query->where('is_read', true);
    }

    public function scopeCritical($query)
    {
        return $query->where('priority', 'critical');
    }

    public function scopeToday($query)
    {
        return $query->whereDate('created_at', Carbon::today());
    }

    public function scopeYesterday($query)
    {
        return $query->whereDate('created_at', Carbon::yesterday());
    }

    public function scopeOlder($query)
    {
        return $query->whereDate('created_at', '<', Carbon::yesterday());
    }

    // Methods
    public function markAsRead(): void
    {
        $this->update([
            'is_read' => true,
            'read_at' => now(),
        ]);
    }

    public function getTimeAgoAttribute(): string
    {
        $created = $this->created_at;
        $now = Carbon::now();

        $diffInMinutes = (int) $created->diffInMinutes($now);
        $diffInHours = (int) $created->diffInHours($now);
        $diffInDays = (int) $created->diffInDays($now);

        if ($diffInMinutes < 1) {
            return 'Just now';
        }
        if ($diffInMinutes < 60) {
            return $diffInMinutes . ' min ago';
        }
        if ($diffInHours < 24) {
            return $diffInHours . ' hour' . ($diffInHours > 1 ? 's' : '') . ' ago';
        }
        if ($diffInDays < 2) {
            return 'Yesterday';
        }

        return $diffInDays . ' days ago';
    }
}
