/**
 * Admin Dashboard - Charts
 */
'use strict';

(function () {
  function initCharts() {
    if (typeof ApexCharts === 'undefined') return;
    const labelColor = (typeof config !== 'undefined' && config.colors) ? config.colors.textMuted : '#a1acb8';
    const borderColor = (typeof config !== 'undefined' && config.colors) ? config.colors.borderColor : '#ebeef0';
    const fontFamily = (typeof config !== 'undefined' && config.fontFamily) ? config.fontFamily : 'inherit';
    const primaryColor = (typeof config !== 'undefined' && config.colors) ? config.colors.primary : '#696cff';

  // Occupancy Donut Chart
  const occupancyChartEl = document.querySelector('#occupancyDonutChart');
  if (occupancyChartEl) {
    const occupied = parseInt(occupancyChartEl.getAttribute('data-occupied') || '14', 10);
    const available = parseInt(occupancyChartEl.getAttribute('data-available') || '10', 10);
    const total = occupied + available;
    const occupiedPct = total ? Math.round((occupied / total) * 100) : 0;

    const occupancyConfig = {
      chart: {
        type: 'donut',
        height: 280,
        fontFamily,
        toolbar: { show: false }
      },
      labels: ['Occupied', 'Available'],
      series: [occupied, available],
      colors: ['#ff6384', '#36a2eb'],
      stroke: { width: 0 },
      dataLabels: {
        enabled: true,
        formatter: function (val) {
          return Math.round(val) + '%';
        }
      },
      legend: {
        show: true,
        position: 'bottom',
        horizontalAlign: 'center',
        labels: { colors: labelColor }
      },
      plotOptions: {
        pie: {
          donut: {
            size: '70%',
            labels: {
              show: true,
              name: { show: true, color: labelColor },
              value: {
                show: true,
                fontSize: '1.5rem',
                fontWeight: 600,
                color: typeof config !== 'undefined' ? config.colors.primary : '#696cff'
              },
              total: {
                show: true,
                label: 'Total Beds',
                formatter: () => total,
                color: labelColor
              }
            }
          }
        }
      },
      grid: { padding: { top: -15, bottom: -15 } }
    };
    new ApexCharts(occupancyChartEl, occupancyConfig).render();
  }

  // Collection vs Expense Trend (Line/Area)
  const trendChartEl = document.querySelector('#collectionTrendChart');
  if (trendChartEl) {
    const labelsJson = trendChartEl.getAttribute('data-labels');
    const collectionJson = trendChartEl.getAttribute('data-collection');
    const expenseJson = trendChartEl.getAttribute('data-expense');
    const labels = labelsJson ? JSON.parse(labelsJson) : ['Aug', 'Sep', 'Oct', 'Nov', 'Dec', 'Jan'];
    const collectionData = collectionJson ? JSON.parse(collectionJson) : [12, 15, 14, 18, 16, 22];
    const expenseData = expenseJson ? JSON.parse(expenseJson) : [8, 9.5, 9, 11, 10, 4.1];

    const trendConfig = {
      chart: {
        type: 'area',
        height: 280,
        fontFamily,
        toolbar: { show: false },
        zoom: { enabled: false },
        stacked: false
      },
      series: [
        { name: 'Collection (₹K)', data: collectionData },
        { name: 'Expense (₹K)', data: expenseData }
      ],
      colors: ['#696cff', '#71dd37'],
      stroke: { curve: 'smooth', width: 2 },
      fill: {
        type: 'gradient',
        gradient: {
          opacityFrom: 0.4,
          opacityTo: 0.05,
          stops: [0, 95]
        }
      },
      xaxis: {
        categories: labels,
        labels: { style: { colors: labelColor, fontSize: '12px' } },
        axisBorder: { show: false },
        axisTicks: { show: false }
      },
      yaxis: {
        labels: {
          style: { colors: labelColor },
          formatter: function (val) {
            return '₹' + val + 'K';
          }
        }
      },
      grid: {
        borderColor,
        strokeDashArray: 6,
        padding: { top: -10, bottom: -5, left: 10, right: 10 }
      },
      legend: {
        position: 'top',
        horizontalAlign: 'right',
        labels: { colors: labelColor }
      },
      tooltip: {
        theme: 'light',
        y: {
          formatter: function (val) {
            return '₹' + val + 'K';
          }
        }
      }
    };
    new ApexCharts(trendChartEl, trendConfig).render();
  }

  // Mini bar chart - Weekly collection (optional, if element exists)
  const weeklyBarEl = document.querySelector('#weeklyCollectionBar');
  if (weeklyBarEl) {
    const weeklyConfig = {
      chart: {
        type: 'bar',
        height: 120,
        toolbar: { show: false },
        sparkline: { enabled: false }
      },
      plotOptions: {
        bar: {
          borderRadius: 4,
          columnWidth: '55%',
          distributed: true
        }
      },
      series: [{ name: 'Collection', data: [4, 5, 3, 6, 5, 4, 7] }],
      colors: ['#696cff', '#696cff', '#696cff', '#696cff', '#696cff', '#696cff', '#71dd37'],
      xaxis: {
        categories: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
        labels: { style: { fontSize: '11px' } }
      },
      yaxis: { labels: { formatter: (v) => '₹' + v + 'K' } },
      grid: {
        borderColor,
        strokeDashArray: 4,
        padding: { top: -10, bottom: -10 }
      },
      dataLabels: { enabled: false }
    };
    new ApexCharts(weeklyBarEl, weeklyConfig).render();
  }
  } // end initCharts

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initCharts);
  } else {
    initCharts();
  }
})();
