@extends('layouts/contentNavbarLayout')

@section('title', 'Drawing')

@section('vendor-style')
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap5.min.css" />
<style>
.drawing-table thead th { font-weight: 600; text-transform: uppercase; font-size: 0.75rem; }
@media print { .no-print { display: none !important; } }
</style>
@endsection

@section('content')
<div class="container-xxl flex-grow-1 container-p-y">
    <div class="d-flex justify-content-between align-items-center flex-wrap gap-2 mb-4">
        <h4 class="fw-bold py-3 mb-0">Drawing</h4>
        <div class="d-flex gap-2 no-print">
            <button type="button" class="btn btn-outline-success btn-sm" id="exportExcel" title="Export Excel"><i class="ri-file-excel-2-line me-1"></i>Excel</button>
            <button type="button" class="btn btn-outline-secondary btn-sm" onclick="window.print();" title="Print"><i class="ri-printer-line me-1"></i>Print</button>
        </div>
    </div>

    @if (session('success'))
        <div class="alert alert-success alert-dismissible" role="alert">
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif
    @if ($errors->any())
        <div class="alert alert-danger alert-dismissible" role="alert">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif

    {{-- Add Drawing --}}
    <div class="card mb-4 no-print">
        <div class="card-header">
            <h5 class="card-title mb-0">Add Drawing</h5>
        </div>
        <div class="card-body">
            <form action="{{ route('admin.drawings.store') }}" method="POST">
                @csrf
                <div class="row g-3">
                    <div class="col-md-2">
                        <label class="form-label">Date <span class="text-danger">*</span></label>
                        <input type="date" class="form-control @error('drawing_date') is-invalid @enderror" name="drawing_date" value="{{ old('drawing_date', date('Y-m-d')) }}" required>
                        @error('drawing_date')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Amount (₹) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control @error('amount') is-invalid @enderror" name="amount" value="{{ old('amount') }}" min="0.01" step="0.01" required>
                        @error('amount')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Type <span class="text-danger">*</span></label>
                        <select class="form-select @error('type') is-invalid @enderror" name="type" required>
                            <option value="">— Select —</option>
                            <option value="credit" {{ old('type') === 'credit' ? 'selected' : '' }}>Credit (funds in)</option>
                            <option value="debit" {{ old('type') === 'debit' ? 'selected' : '' }}>Debit (funds out)</option>
                        </select>
                        @error('type')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Payment Mode <span class="text-danger">*</span></label>
                        <select class="form-select @error('payment_mode') is-invalid @enderror" name="payment_mode" required>
                            <option value="">— Select —</option>
                            @foreach($paymentModes as $key => $label)
                                <option value="{{ $key }}" {{ old('payment_mode') === $key ? 'selected' : '' }}>{{ $label }}</option>
                            @endforeach
                        </select>
                        @error('payment_mode')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">UTR / Cheque No.</label>
                        <input type="text" class="form-control @error('transaction_id') is-invalid @enderror" name="transaction_id" value="{{ old('transaction_id') }}" placeholder="For audit">
                        @error('transaction_id')
                            <div class="invalid-feedback">{{ $message }}</div>
                        @enderror
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Notes</label>
                        <input type="text" class="form-control" name="notes" value="{{ old('notes') }}" placeholder="Optional">
                    </div>
                    <div class="col-md-2 align-self-end">
                        <button type="submit" class="btn btn-primary"><i class="ri-check-line me-1"></i>Save</button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    {{-- Filters --}}
    <div class="card mb-4 no-print">
        <div class="card-body">
            <form method="get" action="{{ route('admin.drawings.index') }}" class="row g-3 align-items-end">
                <div class="col-md-2">
                    <label class="form-label">From Date</label>
                    <input type="date" class="form-control" name="from_date" value="{{ $fromDate ?? '' }}">
                </div>
                <div class="col-md-2">
                    <label class="form-label">To Date</label>
                    <input type="date" class="form-control" name="to_date" value="{{ $toDate ?? '' }}">
                </div>
                <div class="col-md-2">
                    <label class="form-label">Type</label>
                    <select class="form-select" name="type">
                        <option value="">All</option>
                        <option value="credit" {{ ($type ?? '') === 'credit' ? 'selected' : '' }}>Credit</option>
                        <option value="debit" {{ ($type ?? '') === 'debit' ? 'selected' : '' }}>Debit</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Payment Mode</label>
                    <select class="form-select" name="payment_mode">
                        <option value="">All</option>
                        @foreach($paymentModes as $key => $label)
                            <option value="{{ $key }}" {{ ($paymentMode ?? '') === $key ? 'selected' : '' }}>{{ $label }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary">Search</button>
                    <a href="{{ route('admin.drawings.index') }}" class="btn btn-outline-secondary">Reset</a>
                </div>
            </form>
        </div>
    </div>

    {{-- Table --}}
    <div class="card">
        <div class="card-body">
            <div class="table-responsive">
                <table class="table drawing-table table-hover" id="drawingTable">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Type</th>
                            <th>Amount (₹)</th>
                            <th>Payment Mode</th>
                            <th>UTR / Ref.</th>
                            <th>Performed By</th>
                            <th>Notes</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($drawings as $d)
                            <tr>
                                <td>{{ $d->drawing_date->format('d/m/Y') }}</td>
                                <td>{{ $d->type === \App\Models\Drawing::TYPE_CREDIT ? 'Credit' : 'Debit' }}</td>
                                <td>₹ {{ number_format($d->amount, 2) }}</td>
                                <td>{{ $paymentModes[$d->payment_mode] ?? $d->payment_mode }}</td>
                                <td>{{ $d->transaction_id ?? '—' }}</td>
                                <td>{{ $d->performedBy?->name ?? '—' }}</td>
                                <td><span class="text-truncate d-inline-block" style="max-width: 120px;" title="{{ $d->notes }}">{{ $d->notes ?? '—' }}</span></td>
                            </tr>
                        @empty
                            <tr><td colspan="7" class="text-center text-muted py-4">No drawings for the selected filter.</td></tr>
                        @endforelse
                    </tbody>
                    @if($drawings->isNotEmpty())
                    <tfoot>
                        <tr class="table-light fw-bold">
                            <td colspan="2" class="text-end">Credit Total</td>
                            <td>₹ {{ number_format($totalCredit, 2) }}</td>
                            <td></td>
                            <td></td>
                            <td></td>
                            <td></td>
                        </tr>
                        <tr class="table-light fw-bold">
                            <td colspan="2" class="text-end">Debit Total</td>
                            <td>₹ {{ number_format($totalDebit, 2) }}</td>
                            <td></td>
                            <td></td>
                            <td></td>
                            <td></td>
                        </tr>
                        <tr class="table-light fw-bold">
                            <td colspan="2" class="text-end">Net (Credit − Debit)</td>
                            <td>₹ {{ number_format($total, 2) }}</td>
                            <td></td>
                            <td></td>
                            <td></td>
                            <td></td>
                        </tr>
                    </tfoot>
                    @endif
                </table>
            </div>
        </div>
    </div>
</div>
@endsection

@section('vendor-script')
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
@endsection

@section('page-script')
<script>
(function() {
    // Initialize DataTable
    $('#drawingTable').DataTable({
        order: [[0, 'desc']], // Sort by Date descending
        pageLength: 25,
        language: {
            search: "Search:",
            lengthMenu: "Show _MENU_ entries",
            info: "Showing _START_ to _END_ of _TOTAL_ drawings",
            infoEmpty: "No drawings available",
            infoFiltered: "(filtered from _MAX_ total drawings)"
        }
    });

    var btn = document.getElementById('exportExcel');
    if (btn) btn.addEventListener('click', function() {
        var table = document.getElementById('drawingTable');
        if (!table) return;
        var wb = XLSX.utils.table_to_book(table, { sheet: 'Drawings', raw: true });
        XLSX.writeFile(wb, 'drawings-' + new Date().toISOString().slice(0,10) + '.xlsx');
    });
})();
</script>
@endsection
