@extends('layouts/contentNavbarLayout')

@section('title', 'Expenses Details')

@section('vendor-style')
<link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/dataTables.bootstrap5.min.css" />
<style>
.expense-table thead th { font-weight: 600; text-transform: uppercase; font-size: 0.75rem; }
@media print { .no-print { display: none !important; } }
</style>
@endsection

@section('content')
<div class="container-xxl flex-grow-1 container-p-y">
    <div class="d-flex justify-content-between align-items-center flex-wrap gap-2 mb-4">
        <h4 class="fw-bold py-3 mb-0">Expenses Details</h4>
        <div class="d-flex gap-2 no-print">
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addExpenseModal"><i class="ri-add-line me-1"></i>Add Expense</button>
            <a href="{{ route('admin.expense-heads.create') }}" class="btn btn-outline-primary" title="Add Expense Head"><i class="ri-add-line me-1"></i>Add Head</a>
            <button type="button" class="btn btn-outline-success btn-sm" id="exportExcel" title="Export Excel"><i class="ri-file-excel-2-line me-1"></i>Excel</button>
            <button type="button" class="btn btn-outline-secondary btn-sm" onclick="window.print();" title="Print"><i class="ri-printer-line me-1"></i>Print</button>
        </div>
    </div>

    @if (session('success'))
        <div class="alert alert-success alert-dismissible" role="alert">
            {{ session('success') }}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif
    @if ($errors->any())
        <div class="alert alert-danger alert-dismissible" role="alert">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    @endif


    {{-- Filters --}}
    <div class="card mb-4 no-print">
        <div class="card-body">
            <form method="get" action="{{ route('admin.expenses.index') }}" class="row g-3 align-items-end">
                <div class="col-md-2">
                    <label class="form-label">From Date</label>
                    <input type="date" class="form-control" name="from_date" value="{{ $fromDate ?? '' }}">
                </div>
                <div class="col-md-2">
                    <label class="form-label">To Date</label>
                    <input type="date" class="form-control" name="to_date" value="{{ $toDate ?? '' }}">
                </div>
                <div class="col-md-2">
                    <label class="form-label">Head</label>
                    <select class="form-select" name="head_id">
                        <option value="">All</option>
                        @foreach($heads as $h)
                            <option value="{{ $h->id }}" {{ (string)($headId ?? '') === (string)$h->id ? 'selected' : '' }}>{{ $h->name }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Payment Mode</label>
                    <select class="form-select" name="payment_mode">
                        <option value="">All</option>
                        @foreach($paymentModes as $key => $label)
                            <option value="{{ $key }}" {{ ($paymentMode ?? '') === $key ? 'selected' : '' }}>{{ $label }}</option>
                        @endforeach
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary">Search</button>
                    <a href="{{ route('admin.expenses.index') }}" class="btn btn-outline-secondary">Reset</a>
                </div>
            </form>
        </div>
    </div>

    {{-- Table --}}
    <div class="card">
        <div class="card-body">
            <div class="table-responsive">
                <table class="table expense-table table-hover" id="expenseTable">
                    <thead>
                        <tr>
                            <th>Date</th>
                            <th>Head</th>
                            <th>Item</th>
                            <th>Cost (₹)</th>
                            <th>Payment Mode</th>
                            <th>UTR / Ref.</th>
                            <th class="no-print">Bill</th>
                        </tr>
                    </thead>
                    <tbody>
                        @forelse($expenses as $e)
                            <tr>
                                <td>{{ $e->expense_date->format('d/m/Y') }}</td>
                                <td>{{ $e->expenseHead->name ?? '—' }}</td>
                                <td>{{ $e->item_name }}</td>
                                <td>₹ {{ number_format($e->amount, 2) }}</td>
                                <td>{{ $paymentModes[$e->payment_mode] ?? $e->payment_mode }}</td>
                                <td>{{ $e->transaction_id ?? '—' }}</td>
                                <td class="no-print">
                                    @if($e->file_path)
                                        <a href="{{ asset('storage/' . $e->file_path) }}" target="_blank" class="btn btn-sm btn-outline-info" title="View Bill"><i class="ri-file-line"></i></a>
                                    @else
                                        <span class="text-muted">—</span>
                                    @endif
                                </td>
                            </tr>
                        @empty
                            <tr><td colspan="7" class="text-center text-muted py-4">No expenses for the selected filter.</td></tr>
                        @endforelse
                    </tbody>
                    @if($expenses->isNotEmpty())
                    <tfoot>
                        <tr class="table-light fw-bold">
                            <td colspan="3" class="text-end">Total</td>
                            <td>₹ {{ number_format($total, 2) }}</td>
                            <td></td>
                            <td></td>
                            <td class="no-print"></td>
                        </tr>
                    </tfoot>
                    @endif
                </table>
            </div>
        </div>
    </div>
</div>

{{-- Add Expense Modal --}}
<div class="modal fade" id="addExpenseModal" tabindex="-1" aria-labelledby="addExpenseModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addExpenseModalLabel">Add New Expense</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form action="{{ route('admin.expenses.store') }}" method="POST" enctype="multipart/form-data" id="addExpenseForm">
                @csrf
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Date of Expense <span class="text-danger">*</span></label>
                            <input type="date" class="form-control @error('expense_date') is-invalid @enderror" name="expense_date" value="{{ old('expense_date', date('Y-m-d')) }}" required>
                            @error('expense_date')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Expense Head <span class="text-danger">*</span></label>
                            <select class="form-select @error('expense_head_id') is-invalid @enderror" name="expense_head_id" required>
                                <option value="">— Select —</option>
                                @foreach($heads as $h)
                                    <option value="{{ $h->id }}" {{ old('expense_head_id') == $h->id ? 'selected' : '' }}>{{ $h->name }}</option>
                                @endforeach
                            </select>
                            @error('expense_head_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            @if($heads->isEmpty())
                                <p class="form-text text-warning mb-0 mt-1">No heads. <a href="{{ route('admin.expense-heads.create') }}" target="_blank">Add Head</a> first.</p>
                            @endif
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Expense Item <span class="text-danger">*</span></label>
                            <input type="text" class="form-control @error('item_name') is-invalid @enderror" name="item_name" value="{{ old('item_name') }}" placeholder="e.g. 5W LED Bulbs" required>
                            @error('item_name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Cost (₹) <span class="text-danger">*</span></label>
                            <input type="number" class="form-control @error('amount') is-invalid @enderror" name="amount" id="expenseAmount" value="{{ old('amount') }}" min="0" step="0.01" required>
                            @error('amount')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <p class="form-text mb-0 mt-1 small" id="balanceInfo">Select payment mode to see balance.</p>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Payment Mode <span class="text-danger">*</span></label>
                            <select class="form-select @error('payment_mode') is-invalid @enderror" name="payment_mode" id="expensePaymentMode" required>
                                <option value="">— Select —</option>
                                @foreach($paymentModes as $key => $label)
                                    <option value="{{ $key }}" {{ old('payment_mode') === $key ? 'selected' : '' }}>{{ $label }}</option>
                                @endforeach
                            </select>
                            @error('payment_mode')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">UTR / Cheque No.</label>
                            <input type="text" class="form-control" name="transaction_id" value="{{ old('transaction_id') }}" placeholder="For non-cash">
                        </div>
                        <div class="col-12">
                            <label class="form-label">Bill / Receipt (optional)</label>
                            <input type="file" class="form-control @error('bill_file') is-invalid @enderror" name="bill_file" accept=".jpg,.jpeg,.png,.pdf">
                            @error('bill_file')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            <p class="form-text small mb-0">Photo or scan of bill. Max 5MB. JPG, PNG, PDF.</p>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary"><i class="ri-check-line me-1"></i>Save Expense</button>
                </div>
            </form>
        </div>
    </div>
</div>
@endsection

@section('vendor-script')
<script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.7/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
@endsection

@section('page-script')
<script>
(function() {
    // Auto-open modal if there are validation errors
    @if($errors->any())
        var expenseModal = new bootstrap.Modal(document.getElementById('addExpenseModal'));
        expenseModal.show();
    @endif
    
    // Initialize DataTable
    $('#expenseTable').DataTable({
        order: [[0, 'desc']], // Sort by Date descending
        pageLength: 25,
        language: {
            search: "Search:",
            lengthMenu: "Show _MENU_ entries",
            info: "Showing _START_ to _END_ of _TOTAL_ expenses",
            infoEmpty: "No expenses available",
            infoFiltered: "(filtered from _MAX_ total expenses)"
        },
        columnDefs: [
            { orderable: false, targets: -1 } // Disable sorting on last column (Bill)
        ]
    });
    
    var form = document.getElementById('addExpenseForm');
    var modeSelect = document.getElementById('expensePaymentMode');
    var amountInput = document.getElementById('expenseAmount');
    var balanceInfo = document.getElementById('balanceInfo');
    var balanceUrl = '{{ route("admin.expenses.index") }}'.replace(/\/?$/, '') + '/balance';

    function updateBalance() {
        var mode = modeSelect && modeSelect.value;
        if (!mode) {
            balanceInfo.textContent = 'Select payment mode to see balance.';
            return;
        }
        var xhr = new XMLHttpRequest();
        xhr.open('GET', '{{ url(route("admin.expenses.balance")) }}?payment_mode=' + encodeURIComponent(mode));
        xhr.onload = function() {
            if (xhr.status === 200) {
                try {
                    var data = JSON.parse(xhr.responseText);
                    var bal = Number(data.balance) || 0;
                    balanceInfo.textContent = 'Balance: ₹' + bal.toLocaleString('en-IN', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
                } catch (e) {
                    balanceInfo.textContent = 'Could not load balance.';
                }
            } else {
                balanceInfo.textContent = 'Select payment mode to see balance.';
            }
        };
        xhr.onerror = function() { balanceInfo.textContent = 'Select payment mode to see balance.'; };
        xhr.send();
    }
    if (modeSelect) modeSelect.addEventListener('change', updateBalance);
    updateBalance();

    var exportBtn = document.getElementById('exportExcel');
    if (exportBtn) exportBtn.addEventListener('click', function() {
        var table = document.getElementById('expenseTable');
        if (!table) return;
        var wb = XLSX.utils.table_to_book(table, { sheet: 'Expenses', raw: true });
        XLSX.writeFile(wb, 'expenses-' + new Date().toISOString().slice(0,10) + '.xlsx');
    });
})();
</script>
@endsection
